(function(){
  if (!window.wp) return;

  const { element, data, blockEditor, components, hooks, apiFetch, blocks: wpBlocks } = window.wp;
  const { createElement: el, useEffect, useState, useRef } = element;

  // Gutenbergのmeta値は環境/プラグイン干渉/過去の保存形式により
  // 文字列以外（配列・オブジェクト）で返ることがある。
  // そのままString()すると "[object Object]" になり、入力欄が壊れたように見える。
  // できる限り安全に文字列へ正規化する。
  function normalizeMetaToString(v) {
    if (v === null || v === undefined) return '';
    if (typeof v === 'string') return v;
    if (typeof v === 'number' || typeof v === 'boolean') return String(v);

    if (Array.isArray(v)) {
      // 文字列配列なら連結、それ以外は最初の文字列要素を採用
      const strs = v.filter((x) => typeof x === 'string');
      if (strs.length === v.length) return strs.join('');
      if (strs.length > 0) return strs[0];
      return '';
    }

    if (typeof v === 'object') {
      const candidates = ['raw', 'rendered', 'value', 'text', 'content'];
      for (const k of candidates) {
        if (typeof v[k] === 'string') return v[k];
      }

      // まれに {0:'...'} のような形で返るケースへの救済
      for (const key of Object.keys(v)) {
        if (typeof v[key] === 'string') return v[key];
      }
      return '';
    }

    return '';
  }
  const { useSelect, useDispatch, select, dispatch } = data;
  const { BlockControls } = blockEditor;
  const { ToolbarButton, PanelBody, TextareaControl, TextControl, Button, Notice, SelectControl, RangeControl } = components;
  const { addFilter } = hooks;

  // ---- meta keys (template)
  // Meta key is registered in PHP as `ead_master_prompt` for the template CPT.
  const META_MASTER  = 'ead_master_prompt';
  // Trial版：固定フィールドはタイトルのみ
  const FIX_SNIPPET = "【AI_fix:title:】";

  // Trial版：AI_extの挿入UIは提供しない（手打ち/貼り付けは許容）
  const MASTER_PLACEHOLDER = "役割:\nあなたは◯◯専門のWebライターです。\nルール:\n初心者に分かりやすく、具体例を交えて説明してください。";

  // ---- utils
  const escapeRegExp = (s) => s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

  function flattenBlocks(blocks, out=[]) {
    (blocks || []).forEach(b => {
      out.push(b);
      if (b.innerBlocks && b.innerBlocks.length) flattenBlocks(b.innerBlocks, out);
    });
    return out;
  }

  function getSerializedContent(){
    try {
      const blks = select('core/block-editor').getBlocks();
      return (window.wp && window.wp.blocks && typeof window.wp.blocks.serialize === 'function')
        ? window.wp.blocks.serialize(blks)
        : '';
    } catch(e){
      return '';
    }
  }

  // ---- block helpers (same as post)
  function getAllBlocks(){
    return select('core/block-editor').getBlocks();
  }

  function getBlockById(clientId){
    return clientId ? select('core/block-editor').getBlock(clientId) : null;
  }

  function getParents(clientId){
    const store = select('core/block-editor');
    if (store.getBlockParents) return store.getBlockParents(clientId) || [];
    const blocks = getAllBlocks();
    const stack = [];
    let found = false;
    (function walk(list){
      for (const b of list){
        if (b.clientId === clientId){ found = true; return true; }
        if (b.innerBlocks && b.innerBlocks.length){
          stack.push(b.clientId);
          if (walk(b.innerBlocks)) return true;
          stack.pop();
        }
      }
      return false;
    })(blocks);
    return found ? stack.slice() : [];
  }

  function getListAncestorInfo(clientId){
    const parents = getParents(clientId);
    for (let i = parents.length - 1; i >= 0; i--){
      const p = getBlockById(parents[i]);
      if (p && p.name === 'core/list') return { listClientId: p.clientId, parents };
    }
    return null;
  }

  function getNthByPredicate(targetClientId, predicate){
    const flat = flattenBlocks(getAllBlocks(), []);
    let n = 0;
    for (const b of flat){
      if (!predicate(b)) continue;
      n++;
      if (b.clientId === targetClientId) return n;
    }
    return null;
  }

  function getNthParagraph(clientId){
    return getNthByPredicate(clientId, (b)=> b.name === 'core/paragraph');
  }

  function getNthTable(clientId){
    return getNthByPredicate(clientId, (b)=> b.name === 'core/table');
  }

  function getNthHeadingOfLevel(clientId, level){
    return getNthByPredicate(clientId, (b)=> b.name === 'core/heading' && (b.attributes?.level || 2) === level);
  }

  function getNthList(listClientId){
    return getNthByPredicate(listClientId, (b)=> b.name === 'core/list');
  }

  function getListItemIndexWithinList(listClientId, itemClientId){
    const list = getBlockById(listClientId);
    if (!list || !Array.isArray(list.innerBlocks)) return null;
    const items = list.innerBlocks.filter(x => x && x.name === 'core/list-item');
    const idx = items.findIndex(x => x.clientId === itemClientId);
    return idx >= 0 ? idx + 1 : null;
  }

  function parseTableCellFromAttributeKey(attributeKey){
    if (!attributeKey) return null;
    const m = String(attributeKey).match(/^(head|body|foot)\.(\d+)\.cells\.(\d+)\.content$/);
    if (!m) return null;
    return { section: m[1], rowIndex: parseInt(m[2],10), colIndex: parseInt(m[3],10) };
  }

  function colToLetters(n){
    let s = '';
    let x = n;
    while (x >= 0) {
      s = String.fromCharCode((x % 26) + 97) + s;
      x = Math.floor(x / 26) - 1;
    }
    return s;
  }

  function inferTableCellFromDOM(){
    try {
      const el = document.activeElement;
      if (!el) return null;
      const td = el.closest ? el.closest('td,th') : null;
      if (!td || !td.parentElement) return null;
      const tr = td.parentElement;
      const rowIndex = Array.prototype.indexOf.call(tr.parentElement.children, tr);
      const colIndex = Array.prototype.indexOf.call(tr.children, td);
      if (rowIndex < 0 || colIndex < 0) return null;
      return { rowIndex, colIndex };
    } catch(e){
      return null;
    }
  }

  function nextKeyFromBase(baseKey){
    const html = getSerializedContent();
    const re = new RegExp(`【AI:${escapeRegExp(baseKey)}(?:_(\\d+))?:】`, 'g');
    let max = 0;
    let m;
    while ((m = re.exec(html)) !== null) {
      const n = m[1] ? parseInt(m[1], 10) : 1;
      if (n > max) max = n;
    }
    if (max === 0) return baseKey;
    return `${baseKey}_${max + 1}`;
  }

  function getSelectionStartSafe(){
    try {
      const st = select('core/block-editor');
      return st.getSelectionStart ? st.getSelectionStart() : null;
    } catch(e){
      return null;
    }
  }

  function getActiveBlockFromSelection(selStart){
    if (!selStart || !selStart.clientId) return null;
    return getBlockById(selStart.clientId);
  }

  function getCellLabelFromSelection(selStart){
    const parsed = selStart ? parseTableCellFromAttributeKey(selStart.attributeKey) : null;
    if (parsed) {
      const col = colToLetters(parsed.colIndex);
      const row = (parsed.rowIndex + 1);
      return `${col}${row}`;
    }
    const dom = inferTableCellFromDOM();
    if (dom) {
      const col = colToLetters(dom.colIndex);
      const row = (dom.rowIndex + 1);
      return `${col}${row}`;
    }
    return 'a1';
  }

  function buildContextMarker(){
    const selStart = getSelectionStartSafe();
    const b = getActiveBlockFromSelection(selStart);

    // Exclude: math block only
    if (b && (b.name === 'core/math' || b.name === 'core/html')) {
      return null;
    }

    // Global fallback numbering helpers
    const getNextCoreOtherAI = () => {
      const html = getSerializedContent();
      const re = /【AI:AI_(\d+)(?:_\d+)?:】/g;
      let max = 0, m;
      while ((m = re.exec(html)) !== null) {
        const n = parseInt(m[1], 10);
        if (!Number.isNaN(n)) max = Math.max(max, n);
      }
      return `AI_${max + 1}`;
    };

    const getNextNumericOnly = () => {
      const html = getSerializedContent();
      const re = /【AI:(\d+)(?:_\d+)?:】/g;
      let max = 0, m;
      while ((m = re.exec(html)) !== null) {
        const n = parseInt(m[1], 10);
        if (!Number.isNaN(n)) max = Math.max(max, n);
      }
      return String(max + 1);
    };

    if (!b) {
      // If we can't resolve the active block, treat it as "undefined" and use numeric-only keys.
      const base = getNextNumericOnly();
      return `【AI:${base}:】`;
    }

    if (b.name === 'core/table') {
      const tNum = getNthTable(b.clientId) || 1;
      const cell = getCellLabelFromSelection(selStart);
      const base = `t_${tNum}_${cell}`;
      const key = nextKeyFromBase(base);
      return `【AI:${key}:】`;
    }

    if (b.name === 'core/list-item') {
      const info = getListAncestorInfo(b.clientId);
      const listId = info ? info.listClientId : null;
      const lNum = listId ? (getNthList(listId) || 1) : 1;
      const itemNum = listId ? (getListItemIndexWithinList(listId, b.clientId) || 1) : 1;
      const base = `l_${lNum}_${itemNum}`;
      const key = nextKeyFromBase(base);
      return `【AI:${key}:】`;
    }

    if (b.name === 'core/heading') {
      const level = (b.attributes && b.attributes.level) ? b.attributes.level : 2;
      const hNum = getNthHeadingOfLevel(b.clientId, level) || 1;
      const base = `h${level}_${hNum}`;
      const key = nextKeyFromBase(base);
      return `【AI:${key}:】`;
    }

    if (b.name === 'core/paragraph') {
      const pNum = getNthParagraph(b.clientId) || 1;
      const base = `p_${pNum}`;
      const key = nextKeyFromBase(base);
      return `【AI:${key}:】`;
    }

    // Fallback:
    // - WordPress標準ブロック（core/*）のうち、上記で定義されていないもの
    // - WordPress標準以外（非 core/*）の未定義ブロック
    // どちらも【AI:N:】の数字のみで自動採番
    const base = getNextNumericOnly();
    return `【AI:${base}:】`;
  }

  // ---- Caret tracking (selectionchange)
  let __eadLastRange = null;
  let __eadLastRich = null;

  function __eadCaptureCaretRange(){
    const sel = window.getSelection ? window.getSelection() : null;
    if (!sel || !sel.rangeCount) return;
    const range = sel.getRangeAt(0);
    const container = range.commonAncestorContainer;
    const elNode = container && container.nodeType === 1 ? container : (container ? container.parentElement : null);
    if (!elNode || !elNode.closest) return;
    const rich = elNode.closest('[contenteditable="true"]');
    if (!rich) return;
    const inEditor = rich.closest('.edit-post-visual-editor, .block-editor-writing-flow, .block-editor-block-list__layout, .editor-styles-wrapper, .block-editor');
    if (!inEditor) return;
    try {
      __eadLastRange = range.cloneRange();
      __eadLastRich = rich;
    } catch (e) {}
  }

  try {
    document.addEventListener('selectionchange', __eadCaptureCaretRange, true);
  } catch (e) {}

  function insertMarkerAtCaret(event){
    try { if (event && event.preventDefault) event.preventDefault(); } catch(e) {}

    __eadCaptureCaretRange();

    const marker = buildContextMarker();
    if (!marker) return;

    // Trial版：本文スロット上限（物理制限）
    // - key種別（h2_1 / p_2 / t_1_a1 / 数字のみ など）を問わず、【AI: ... :】を「本文スロット」として数える
    // - 過去の正規表現は _数字 を含むものしか数えず、上限が効かないケースがあり得た
    const MAX_SLOTS = 10;
    const content = (wp.data.select('core/editor').getEditedPostContent && wp.data.select('core/editor').getEditedPostContent()) || '';
    const existingSlots = (content.match(/【AI[:：]/g) || []).length;
    if (existingSlots >= MAX_SLOTS) {
      alert(`本文スロットは最大 ${MAX_SLOTS} 個までです。`);
      return;
    }

    const sel = window.getSelection ? window.getSelection() : null;
    if (!sel) return;

    if (__eadLastRange) {
      try {
        sel.removeAllRanges();
        sel.addRange(__eadLastRange);
      } catch (e) {}
    }

    if (!sel.rangeCount) return;

    const range = sel.getRangeAt(0);
    const container = range.commonAncestorContainer;
    const elNode = container && container.nodeType === 1 ? container : (container ? container.parentElement : null);
    const rich = __eadLastRich || (elNode && elNode.closest ? elNode.closest('[contenteditable="true"]') : null);
    if (!rich) return;

    const inEditor = rich.closest('.edit-post-visual-editor, .block-editor-writing-flow, .block-editor-block-list__layout, .editor-styles-wrapper, .block-editor');
    if (!inEditor) return;

    try { rich.focus(); } catch(e) {}

    let ok = false;
    try {
      ok = document.execCommand('insertText', false, marker);
    } catch(e) {
      ok = false;
    }

    if (!ok) {
      try {
        range.deleteContents();
        range.insertNode(document.createTextNode(marker));
        range.collapse(false);
        sel.removeAllRanges();
        sel.addRange(range);
      } catch(e2) {
        return;
      }
      try {
        rich.dispatchEvent(new InputEvent('input', { bubbles: true }));
      } catch(e3) {
        try { rich.dispatchEvent(new Event('input', { bubbles: true })); } catch(e4) {}
      }
    }

    __eadCaptureCaretRange();
  }

  // ---- Toolbar injection (same behavior as post)
  const AI_ICON = el('svg', { width: 20, height: 20, viewBox: '0 0 20 20', xmlns: 'http://www.w3.org/2000/svg' },
    el('rect', { x: 1.5, y: 1.5, width: 17, height: 17, rx: 3, ry: 3, fill: 'none', stroke: 'currentColor', strokeWidth: 1.5 }),
    el('text', { x: 10, y: 13, textAnchor: 'middle', fontSize: 9.5, fontFamily: 'system-ui, -apple-system, Segoe UI, Roboto, sans-serif', fill: 'currentColor' }, 'AI')
  );

  const withAIToolbar = (BlockEdit) => (props) => {
    if (!props || !props.isSelected) return el(BlockEdit, props);

    useSelect((sel) => {
      const st = sel('core/block-editor');
      return { selectionStart: st.getSelectionStart ? st.getSelectionStart() : null };
    }, [props.clientId]);

    const onMouseDown = (event) => {
      try { event.preventDefault(); } catch (e) {}
      insertMarkerAtCaret(event);
    };

    return el(
      element.Fragment,
      {},
      el(BlockEdit, props),
      el(
        BlockControls,
        { group: 'block' },
        el(ToolbarButton, {
          icon: AI_ICON,
          label: 'AIスロット',
          onMouseDown,
        })
      )
    );
  };

  addFilter('editor.BlockEdit', 'ead/ai-toolbar-template', withAIToolbar);

  // ---- Sidebar UI (template)
  function isEditableTextBlock(b){
    if (!b) return false;
    // For template generation, we'll target blocks that contain AI markers in text.
    // Table is special and can contain markers inside cells.
    if (b.name === 'core/table') return true;
    // For headings/paragraphs/list-items, markers live in text content (serialized HTML).
    if (b.name === 'core/paragraph') return true;
    if (b.name === 'core/heading') return true;
    if (b.name === 'core/list-item') return true;
    return false;
  }

  function SidebarApp(){
    // Meta is loaded asynchronously by the editor.
    // If we freeze the selector with an empty deps array, saved meta values
    // (e.g. template master prompt) may not appear when reopening.
    const meta = useSelect((sel)=> sel('core/editor').getEditedPostAttribute('meta') || {});
    const editPost = useDispatch('core/editor').editPost;

    const setMetaValue = (key, value) => {
      const currentMeta = meta || {};
      editPost({ meta: { ...currentMeta, [key]: value } });
    };

// ---- Master prompt sync (align with post editor: immediate meta sync + debounced ajax save)
const postId = useSelect((sel)=> sel('core/editor').getCurrentPostId());

const resolvePostId = () => {
  // getCurrentPostId() can be null transiently; fall back to the edited post id or URL.
  try {
    const coreEditor = (window.wp && window.wp.data && window.wp.data.select) ? window.wp.data.select('core/editor') : null;
    const pidFromStore = coreEditor && typeof coreEditor.getCurrentPostId === 'function'
      ? coreEditor.getCurrentPostId()
      : null;
    const pidFromEdited = coreEditor && typeof coreEditor.getEditedPostAttribute === 'function'
      ? coreEditor.getEditedPostAttribute('id')
      : null;

    let pid = postId || pidFromStore || pidFromEdited;

    if (!pid && typeof window !== 'undefined') {
      const p = new URLSearchParams(window.location.search).get('post');
      pid = p ? parseInt(p, 10) : null;
    }

    const n = parseInt(String(pid || ''), 10);
    return Number.isFinite(n) && n > 0 ? n : null;
  } catch (e) {
    return null;
  }
};

const wpAjaxGetMeta = async (pid, metaKeys) => {
  const body = new URLSearchParams();
  body.set('action', 'ead_get_meta');
  body.set('nonce', (window.EAD && window.EAD.ajax_nonce) ? window.EAD.ajax_nonce : '');
  body.set('post_id', String(pid || ''));
  body.set('meta_keys', JSON.stringify(metaKeys || []));
  const res = await fetch((window.EAD && window.EAD.ajax_url) ? window.EAD.ajax_url : ajaxurl, {
    method: 'POST',
    credentials: 'same-origin',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
    body: body.toString(),
  });
  const json = await res.json();
  if (!json || !json.success) throw new Error((json && json.data && json.data.message) ? json.data.message : 'AJAX get_meta failed');
  return json.data && json.data.meta ? json.data.meta : {};
};

const wpAjaxSaveMeta = async (pid, metaKey, metaValue) => {
  const body = new URLSearchParams();
  body.set('action', 'ead_save_meta');
  body.set('nonce', (window.EAD && window.EAD.ajax_nonce) ? window.EAD.ajax_nonce : '');
  body.set('post_id', String(pid || ''));
  body.set('meta_key', String(metaKey || ''));
  body.set('meta_value', String(metaValue || ''));
  const res = await fetch((window.EAD && window.EAD.ajax_url) ? window.EAD.ajax_url : ajaxurl, {
    method: 'POST',
    credentials: 'same-origin',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
    body: body.toString(),
  });
  const json = await res.json();
  if (!json || !json.success) throw new Error((json && json.data && json.data.message) ? json.data.message : 'AJAX save_meta failed');
  return true;
};

const syncLegacyCustomFieldsMetabox = (metaKey, nextValue) => {
  // If "Custom fields" metabox is enabled, its submitted value can override REST meta.
  // Keep the existing row's textarea value synced (do NOT add a new meta row).
  try {
    const metabox = document.getElementById('postcustom');
    if (!metabox) return;

    const list = metabox.querySelector('#the-list');
    if (!list) return;

    const rows = Array.from(list.querySelectorAll('tr'));
    let targetRow = null;

    for (const tr of rows) {
      const keyInput = tr.querySelector('input[name$="[key]"]');
      const key = keyInput ? String(keyInput.value || '') : '';
      if (key === String(metaKey)) {
        targetRow = tr;
        break;
      }
    }

    if (!targetRow) return;

    const valueTextarea = targetRow.querySelector('textarea[name$="[value]"]');
    if (!valueTextarea) return;

    valueTextarea.value = String(nextValue || '');
    try { valueTextarea.dispatchEvent(new Event('input', { bubbles: true })); } catch(e) {}
  } catch (e) {
    // ignore
  }
};

// Master prompt from Gutenberg meta store.
// NOTE: Gutenberg may transiently report empty/stale meta during save or sidebar remount.
// We must avoid a "watch & overwrite" loop, otherwise user input gets clobbered.
const masterPromptMeta = normalizeMetaToString(meta[META_MASTER]);
const hasMasterMetaKey = Object.prototype.hasOwnProperty.call((meta || {}), META_MASTER);
const masterUserEditedRef = useRef(false);
const masterDidInitRef = useRef(false);
const masterSaveTimerRef = useRef(null);
const masterLastValueRef = useRef('');

const ensureMasterCache = () => {
  try {
    const pid = resolvePostId();
    if (!pid) return null;
    const w = window;
    w.__EAD_MASTER_PROMPT_CACHE = w.__EAD_MASTER_PROMPT_CACHE || {};
	    if (!w.__EAD_MASTER_PROMPT_CACHE[pid]) {
	      w.__EAD_MASTER_PROMPT_CACHE[pid] = { edited: false, value: String(masterPromptMeta || '') };
	    }
    // If the cache says "edited", treat as editing even if this component remounted.
    if (w.__EAD_MASTER_PROMPT_CACHE[pid].edited) {
      masterUserEditedRef.current = true;
    }
    return w.__EAD_MASTER_PROMPT_CACHE[pid];
  } catch (e) {
    return null;
  }
};

const [masterPrompt, _setMasterPrompt] = useState(() => {
  const c = ensureMasterCache();
  return c ? String(c.value || '') : String(masterPromptMeta || '');
});

const setMasterPromptValue = (v) => {
  const next = String(v || '');
  masterUserEditedRef.current = true;
  const c = ensureMasterCache();
  if (c) { c.edited = true; c.value = next; }
  // Update local UI state immediately (controlled textarea)
  _setMasterPrompt(next);

  masterLastValueRef.current = next;

  // Immediate sync to Gutenberg meta store (like post editor)
  setMetaValue(META_MASTER, next);

  // Keep legacy Custom Fields metabox in sync to avoid overwriting on save
  syncLegacyCustomFieldsMetabox(META_MASTER, next);

  // Debounced persistence via admin-ajax.
  if (masterSaveTimerRef.current) clearTimeout(masterSaveTimerRef.current);
  masterSaveTimerRef.current = setTimeout(async () => {
    try {
      const pid = resolvePostId();
      if (!pid) return;


      // If this timer is stale (user typed again), do nothing.
      if (masterLastValueRef.current !== next) return;

      await wpAjaxSaveMeta(pid, META_MASTER, next);
    } catch (e) {
      // ignore
    }
  }, 400);
};

// Clear pending master save timer on unmount
useEffect(() => {
  return () => {
    if (masterSaveTimerRef.current) {
      try { clearTimeout(masterSaveTimerRef.current); } catch(e) {}
      masterSaveTimerRef.current = null;
    }
  };
}, []);

// Initial load hydration only when the user has not edited and local is still empty.
// This matches editor-post.js behavior: do not clobber user input (e.g. leading blank lines) during typing.
useEffect(() => {
  if (masterUserEditedRef.current) return;
  if (!hasMasterMetaKey) return;
  const incoming = String(masterPromptMeta || '');
  if (masterPrompt === '' && incoming !== '') {
    const c = ensureMasterCache();
    _setMasterPrompt(incoming);
    if (c) c.value = incoming;
  }
}, [hasMasterMetaKey, masterPromptMeta]);

const master = masterPrompt;

    const [topic, setTopic] = useState('');
    const [genStatus, setGenStatus] = useState('');
    const [slotInfo, setSlotInfo] = useState(null);
    const [aiProvider, setAiProvider] = useState('gemini');
    const [aiModel, setAiModel] = useState('gemini-2.5-flash');
    const [creativity, setCreativity] = useState(40);

    // load default AI settings (from admin settings)
    useEffect(() => {
      (async () => {
        try {
          const res = await apiFetch({ path: '/ead/v1/settings' });
          const d = res && res.defaults ? res.defaults : null;
          if (d) {
            if (d.ai_provider) setAiProvider(d.ai_provider);
            if (d.ai_model) setAiModel(d.ai_model);
            if (typeof d.creativity !== 'undefined') setCreativity(parseInt(d.creativity, 10) || 0);
          }
        } catch (e) {
          // ignore
        }
      })();
    }, []);

    const appendToMaster = (snippet) => {
      const next = (master || '') + snippet;
      setMasterPromptValue(next);
    };

    const insertFix = () => {
      appendToMaster(FIX_SNIPPET);
    };

    // Trial版：AI_ext挿入ボタンは廃止

    const doSlotCheck = () => {
      try {
        // スロットチェックは「記事本文（post_content）」のみを対象にする。
        // マスタープロンプト／更新プロンプト等のメタは含めない。
        const all = String(select('core/editor').getEditedPostContent?.() || '');

        const openRe = /【(AI|AI_fix|AI_ext):/g;
        const slotRe = /【(AI|AI_fix|AI_ext):([^:】]+):[^】]*】/g;

        const openCount = (all.match(openRe) || []).length;
        const slotMatches = all.match(slotRe) || [];
        const closedCount = slotMatches.length;
        const unclosedAI = Math.max(0, openCount - closedCount);

        // 重複キーは type:key で判定する（AI / AI_fix / AI_ext を区別）
        const keys = [];
        const slotRe2 = /【(AI|AI_fix|AI_ext):([^:】]+):[^】]*】/g;
        let m;
        while ((m = slotRe2.exec(all)) !== null) {
          keys.push(`${m[1]}:${m[2]}`);
        }
        const map = {};
        keys.forEach((k) => { map[k] = (map[k] || 0) + 1; });
        const dups = Object.keys(map).filter((k) => map[k] > 1);

	    // {{topic}} のみスロットとして数える（全角｛｛｝｝は除外）
	    const topicRe = /\{\{topic\}\}/g;
        const topicCount = (all.match(topicRe) || []).length;

        // 閉じ忘れ検出（簡易）
        const asciiOpen = (all.match(/\{\{/g) || []).length;
        const asciiClose = (all.match(/\}\}/g) || []).length;
        const fwOpen = (all.match(/｛｛/g) || []).length;
        const fwClose = (all.match(/｝｝/g) || []).length;
        const unclosedTopic = Math.max(0, asciiOpen - asciiClose) + Math.max(0, fwOpen - fwClose);

        setSlotInfo({ total: slotMatches.length + topicCount, dupKeys: dups, unclosed: unclosedAI + unclosedTopic });
      } catch (e) {
        setSlotInfo({ error: e?.message || String(e) });
      }
    };

    const [busy, setBusy] = useState(false);
    const [error, setError] = useState('');

    const doGenerate = async () => {
      setBusy(true); setError('');
      setGenStatus('AIに送信中…');
      try{
        const postId = select('core/editor').getCurrentPostId();
        const blocks = select('core/block-editor').getBlocks();
        const contextHtml = (window.wp && window.wp.blocks && typeof window.wp.blocks.serialize === 'function')
          ? window.wp.blocks.serialize(blocks)
          : '';

        // Collect targets: only blocks that contain AI markers (or tables).
        const flat = flattenBlocks(blocks, []);
        const aiRe = /【AI:([^:】]+):[^】]*】/g;
        const targets = [];

        flat.forEach(b => {
          if (!isEditableTextBlock(b)) return;
          if (b.name === 'core/table') {
            // For tables, include all cells that have markers.
            const attrs = b.attributes || {};
            ['head','body','foot'].forEach(section => {
              const rows = attrs[section];
              if (!Array.isArray(rows)) return;
              rows.forEach((row, rIdx) => {
                if (!row || !Array.isArray(row.cells)) return;
                row.cells.forEach((cell, cIdx) => {
                  const html = (cell && typeof cell.content === 'string') ? cell.content : '';
                  aiRe.lastIndex = 0;
                  const keys = [];
                  let m;
                  while ((m = aiRe.exec(html)) !== null) keys.push(m[1]);
                  if (!keys.length) return;
                  targets.push({
                    clientId: b.clientId,
                    blockName: b.name,
                    primary: `${section}.${rIdx}.cells.${cIdx}.content`,
                    keys,
                  });
                });
              });
            });
            return;
          }

          // Non-table blocks: serialize and extract keys from common text areas
          const html = (window.wp && window.wp.blocks && typeof window.wp.blocks.getBlockContent === 'function')
            ? window.wp.blocks.getBlockContent(b)
            : '';
          aiRe.lastIndex = 0;
          const keys = [];
          let m;
          while ((m = aiRe.exec(html)) !== null) keys.push(m[1]);
          if (!keys.length) return;
          targets.push({
            clientId: b.clientId,
            blockName: b.name,
            primary: 'content',
            keys,
          });
        });

        setGenStatus('生成中…');
        const res = await apiFetch({
          path: '/ead/v1/generate_template',
          method: 'POST',
          data: {
            post_id: postId,
            mode: 'template',
            master_instruction: master,
            topic: topic,
            context_html: contextHtml,
            targets_json: JSON.stringify(targets),
            ai_provider: aiProvider,
            ai_model: aiModel,
            creativity: creativity,
          }
        });

        if (res && res.success && res.blocks) {
          // Apply results (best-effort): update block attributes by clientId.
          res.blocks.forEach((t) => {
            if (!t || !t.clientId || !t.primary) return;
            const b = select('core/block-editor').getBlock(t.clientId);
            if (!b) return;
            const attrs = b.attributes || {};

            // Table cell path update
            if (b.name === 'core/table' && typeof t.primary === 'string' && t.primary.indexOf('cells') !== -1) {
              const parts = t.primary.split('.');
              const section = parts[0];
              const rIdx = parseInt(parts[1], 10);
              const cIdx = parseInt(parts[3], 10);
              const rows = attrs[section];
              if (!Array.isArray(rows) || !rows[rIdx] || !Array.isArray(rows[rIdx].cells) || !rows[rIdx].cells[cIdx]) return;
              const rows2 = rows.slice();
              rows2[rIdx] = Object.assign({}, rows2[rIdx], { cells: rows2[rIdx].cells.slice() });
              rows2[rIdx].cells[cIdx] = Object.assign({}, rows2[rIdx].cells[cIdx], { content: String(t.value || '') });
              const patch = {}; patch[section] = rows2;
              dispatch('core/block-editor').updateBlockAttributes(t.clientId, patch);
              return;
            }

            // Default: update content
            const patch = { content: String(t.value || '') };
            dispatch('core/block-editor').updateBlockAttributes(t.clientId, patch);
          });

          if (res && res.edit_link) { window.location.href = res.edit_link; return; }
          setGenStatus('');
          dispatch('core/notices').createNotice('success', '下書き生成が完了しました', { isDismissible: true });
        } else {
          throw new Error(res && res.message ? res.message : '生成に失敗しました');
        }
      }catch(e){
        setGenStatus(`エラー（${e?.message || String(e)}）`);
        setError(e?.message || String(e));
      }finally{
        setBusy(false);
      }
    };

    return el(
      'div',
      { style: { padding: '12px' } },
      error ? el(Notice, { status: 'error', onRemove: ()=>setError('') }, error) : null,

      // --- AI指示 ---
      el(
        PanelBody,
        { title: 'AI指示', initialOpen: true },

        el('p', { style: { fontSize: '12px', color: '#666', marginTop: '0' } },
          'どんな役割・ルールで文章を当てはめるかを記述します。'
        ),

        el('div', { style: { marginTop: '8px' } },
          el('div', { style: { fontWeight: '600', marginBottom: '6px' } }, 'マスタープロンプト'),
                    el('textarea', {
            value: masterPrompt,
            placeholder: MASTER_PLACEHOLDER,
            onChange: (e)=> {
              const v = (e && e.target) ? e.target.value : e;
              setMasterPromptValue(v);
            },
            onInput: (e)=> {
              // auto-grow
              if (e && e.target) {
                e.target.style.height = 'auto';
                e.target.style.height = (e.target.scrollHeight + 2) + 'px';
              }
            },
            onFocus: (e)=> {
              if (!e || !e.target) return;
              e.target.style.height = 'auto';
              e.target.style.height = (e.target.scrollHeight + 2) + 'px';
            },
            style: {
              width: '100%',
              boxSizing: 'border-box',
              padding: '8px',
              borderRadius: '6px',
              border: '1px solid #ccd0d4',
              minHeight: '160px',
              resize: 'none',
              lineHeight: '1.6',
            },
          })
        ),

        el('div', { style: { borderTop: '1px solid #eee', margin: '12px 0' } }),

        el('div', {},
          el('p', { style: { fontSize: '12px', color: '#666', margin: '0 0 6px 0' } },
            '固定フィールド（タイトル）にAIで値を生成したい場合に使用します。指示は「:】」の間に記述してください。'
          ),
          el(Button, { variant: 'secondary', onClick: insertFix }, 'タイトル（AI_fix:title）挿入'),
        ),

        el('div', { style: { borderTop: '1px solid #eee', margin: '12px 0' } }),

        el('div', {},
          el('p', { style: { fontSize: '12px', color: '#666', margin: '0 0 6px 0' } },
            'スロットの状態を簡易チェックします（重複や閉じ忘れがあっても生成は可能です）。'
          ),
          el(Button, { variant: 'secondary', onClick: doSlotCheck }, 'スロットチェック'),
          slotInfo ? el('div', { style: { marginTop: '8px', fontSize: '12px' } },
            slotInfo.error ? el('div', { style: { color: '#b32d2e' } }, `エラー（${slotInfo.error}）`) : null,
            !slotInfo.error ? el('div', {}, `スロット総数：${slotInfo.total}`) : null,
            !slotInfo.error ? el('div', {}, `キー重複：${slotInfo.dupKeys && slotInfo.dupKeys.length ? 'あり（' + slotInfo.dupKeys.join(', ') + '）' : 'なし'}`) : null,
            !slotInfo.error ? el('div', {}, `閉じ忘れ検出：${slotInfo.unclosed ? 'あり（' + slotInfo.unclosed + '）' : 'なし'}`) : null,
          ) : null,
        ),
      ),

      // --- AI記事生成 ---
      el(
        PanelBody,
        { title: 'AI記事生成', initialOpen: true },

        el('p', { style: { fontSize: '12px', color: '#666', marginTop: '0' } },
          '生成する記事の題材（トピック）を入力してください。本文中に {{topic}} を置くと、入力したトピックに差し替えられます。'
        ),
        el(TextControl, {
          
          __next40pxDefaultSize: true,
          __nextHasNoMarginBottom: true,label: 'トピック {{topic}}（保存なし）',
          value: topic,
          onChange: (v)=> setTopic(v),
        }),

        el('div', { style: { fontSize: '12px', color: '#666', marginTop: '8px' } },
          '生成された内容は必ず確認し、必要に応じて修正してください。'
        ),

        el(Button, {
          variant: 'primary',
          isBusy: busy,
          disabled: busy || !master,
          onMouseDown: (e)=>e.preventDefault(),
          onClick: doGenerate,
          // 幅を文字量に合わせ、角丸はWP標準に合わせる
          style: { marginTop: '8px', width: 'fit-content' }
        }, '下書き生成'),

        genStatus ? el('div', { style: { marginTop: '8px', fontSize: '12px', color: '#666' } }, genStatus) : null,
      ),

      // --- AI設定 ---
      el(
        PanelBody,
        { title: 'AI設定', initialOpen: false },

        el(SelectControl, {
          label: 'AIの選択',
          value: aiProvider,
          options: [
            { label: 'Gemini', value: 'gemini' },
          ],
          onChange: (v)=> setAiProvider(v),
        }),

        el(SelectControl, {
          label: 'モデル選択',
          value: aiModel,
          options: (aiProvider === 'gemini') ? [
            { label: 'gemini-2.5-flash', value: 'gemini-2.5-flash' },
            { label: 'gemini-2.5-pro', value: 'gemini-2.5-pro' },
          ] : [],
          onChange: (v)=> setAiModel(v),
        }),

        el('div', { style: { marginTop: '10px' } },
          el('div', { style: { fontWeight: '600', marginBottom: '6px' } }, '文章の自由度'),
          el('div', { style: { display: 'flex', gap: '10px', alignItems: 'center' } },
            el('div', { style: { flex: '1' } },
              el(RangeControl, {
                value: creativity,
                onChange: (v)=> setCreativity(parseInt(v,10) || 0),
                min: 0,
                max: 100,
              })
            ),
          ),
          el('div', { style: { fontSize: '12px', color: '#666', marginTop: '4px' } },
            '0=指示に忠実 / 100=自由に表現（まずは35前後がおすすめ）'
          )
        ),
      )
    );
  }

  if (window.wp.editor && window.wp.editor.PluginSidebar && window.wp.plugins) {
    const { registerPlugin } = window.wp.plugins;
    const { PluginSidebar, PluginSidebarMoreMenuItem } = window.wp.editor;

    registerPlugin('ead-template', {
      render: function() {
        return el(
          element.Fragment,
          {},
          el(PluginSidebarMoreMenuItem, {
            target: 'ead-template-sidebar',
            icon: AI_ICON,
          }, 'Easy AI Director'),
          el(PluginSidebar, {
            name: 'ead-template-sidebar',
            title: 'Easy AI Director',
            icon: AI_ICON,
          }, el(SidebarApp))
        );
      }
    });
  }
})()