<?php
/**
 * Plugin Name:       Easy AI Director (Trial)
 * Plugin URI:        https://easyaidirector.com/
 * Description:       Gutenberg上で「スロット」を指定してAI下書きを生成し、構成〜執筆を効率化するプラグイン（体験版）。体験版ではテンプレート数とスロット数に上限があります。
 * Version:           0.3.21-trial
 * Requires at least: 6.0
 * Requires PHP:      7.4
 * Author:            hisa
 * Author URI:        https://easyaidirector.com/
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       easy-ai-director
 */
if ( ! defined( 'ABSPATH' ) ) exit;

// JS/CSS のキャッシュバスター等に使用
if ( ! defined( 'EAD_VERSION' ) ) {
  define('EAD_VERSION', '0.3.21-trial');
	// ↑ assetsのキャッシュ更新用（上のプラグインVersionと合わせる）
	// NOTE: Version bumped in header; keep this in sync.
	
	// ↑ assetsのキャッシュ更新用（上のプラグインVersionと合わせる）
}

require_once __DIR__ . '/includes/settings.php';
require_once __DIR__ . '/includes/rest-generate-template.php';
// Trial版：更新生成は提供しない（RESTもUIも含めない）
// require_once __DIR__ . '/includes/rest-generate-update.php';

class Easy_AI_Director {
    const TEMPLATE_CPT = 'ead_template';

    const META_MASTER  = 'ead_master_prompt';
    const META_TOPIC   = 'ead_topic';
    // Trial版では更新プロンプト/履歴は使わない
    // const META_UPDATE  = 'ead_update_prompt';
    // const META_HISTORY = 'ead_update_history';

    /**
     * Sanitizers must accept 3 args because WordPress calls them as:
     * sanitize_callback( $value, $meta_key, $object_type ).
     * Passing core sanitizers directly can trigger ArgumentCountError in PHP 8+.
     */
    public static function sanitize_textarea_3($value, $meta_key = '', $object_type = '') {
        return sanitize_textarea_field((string) $value);
    }

    public static function sanitize_text_3($value, $meta_key = '', $object_type = '') {
        return sanitize_text_field((string) $value);
    }

    public static function sanitize_history_3($value, $meta_key = '', $object_type = '') {
        return wp_kses_post((string) $value);
    }

    public static function init() {
        add_action('init', [__CLASS__, 'register_template_cpt'], 10);
        add_action('init', [__CLASS__, 'register_meta_fields'], 20);

        add_action('enqueue_block_editor_assets', [__CLASS__, 'enqueue_editor_assets']);

        // Trial版：テンプレ上限(5件)に達したら「新規追加」ボタンを隠す
        add_action('admin_head', [__CLASS__, 'maybe_hide_add_new_template_button']);

        // Fallback for environments where WP REST API routes are disabled/rewritten (REST 404/401).
        // We persist essential draft meta via admin-ajax so reopening the editor restores the value.
        add_action('wp_ajax_ead_save_meta', [__CLASS__, 'ajax_save_meta']);
        add_action('wp_ajax_ead_get_meta', [__CLASS__, 'ajax_get_meta']);

        add_action('save_post', [__CLASS__, 'maybe_apply_ext_json'], 10, 3);

        register_activation_hook(__FILE__, [__CLASS__, 'on_activate']);
    }

    private static function ajax_verify_request($post_id) {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!$nonce || !wp_verify_nonce($nonce, 'ead_ajax')) {
            wp_send_json_error(['message' => 'bad nonce'], 403);
        }
        if (!$post_id || !current_user_can('edit_post', $post_id)) {
            wp_send_json_error(['message' => 'forbidden'], 403);
        }
    }

    public static function ajax_save_meta() {
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        self::ajax_verify_request($post_id);

        $key = isset($_POST['meta_key']) ? sanitize_key(wp_unslash($_POST['meta_key'])) : '';
        // Allow EAD meta keys + AI_ext arbitrary keys (no leading underscore)
        if (!$key || strpos($key, '_') === 0) {
            wp_send_json_error(['message' => 'invalid meta_key'], 400);
        }


		// JS may send either 'value' or 'meta_value'. Accept both.
		if (isset($_POST['value'])) {
			$value = wp_unslash($_POST['value']);
		} elseif (isset($_POST['meta_value'])) {
			$value = wp_unslash($_POST['meta_value']);
		} else {
			$value = '';
		}
        if (!is_string($value)) $value = '';

        if ($key === self::META_TOPIC) {
            $value = sanitize_text_field($value);
        } else {
            $value = sanitize_textarea_field((string)$value);
        }

        // Enforce single-row meta to avoid duplicated custom fields (e.g. ead_master_prompt).
        delete_post_meta($post_id, $key);
        add_post_meta($post_id, $key, $value, true);
        wp_send_json_success(['saved' => true, 'meta_key' => $key, 'value' => $value]);
    }

    public static function ajax_get_meta() {
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        self::ajax_verify_request($post_id);

        // If duplicated custom fields exist (e.g. multiple ead_master_prompt),
        // WordPress may return an empty first value. Pick the last non-empty value and
        // consolidate into a single meta row.
        $keys = [ self::META_MASTER, self::META_TOPIC ];
        $out = [];
        foreach ($keys as $k) {
            $vals = get_post_meta($post_id, $k, false);
            $best = '';
            if (is_array($vals)) {
                foreach ($vals as $v) {
                    $sv = is_scalar($v) ? (string) $v : '';
                    if (trim($sv) !== '') {
                        $best = $sv;
                    }
                }
                if ($best === '' && count($vals) > 0) {
                    $best = is_scalar(end($vals)) ? (string) end($vals) : '';
                }
                if (count($vals) > 1) {
                    delete_post_meta($post_id, $k);
                    add_post_meta($post_id, $k, $best, true);
                }
            }
            $out[$k] = (string) $best;
        }

        wp_send_json_success(['meta' => $out]);
    }

    public static function on_activate() {
        self::register_template_cpt();
        flush_rewrite_rules();
    }

    public static function register_template_cpt() {
        register_post_type(self::TEMPLATE_CPT, [
            'labels'        => [
                'name'      => 'テンプレート',
                'menu_name' => 'テンプレート',
            ],
            'public'        => false,
            'show_ui'       => true,
            'show_in_menu'  => false,
            'show_in_rest'  => true,
            'menu_position' => 25,
            'menu_icon'     => 'dashicons-media-text',
            'supports'      => ['title', 'editor', 'revisions', 'custom-fields'],
            'rewrite'       => false,
            'has_archive'   => false,
        ]);
    }

    public static function register_meta_fields() {
        $auth = function() { return current_user_can('edit_posts'); };

        $rest_schema = [
            'schema' => [
                'type'    => 'string',
                'default' => '',
                'context' => ['view', 'edit'],
            ],
        ];


        // template
        register_post_meta(self::TEMPLATE_CPT, self::META_MASTER, [
            'type'          => 'string',
            'single'        => true,
            'default'       => '',
            'show_in_rest'  => $rest_schema,
            'auth_callback' => $auth,
            'sanitize_callback' => [__CLASS__, 'sanitize_textarea_3'],
        ]);

        register_post_meta(self::TEMPLATE_CPT, self::META_TOPIC, [
            'type'          => 'string',
            'single'        => true,
            'default'       => '',
            'show_in_rest'  => $rest_schema,
            'auth_callback' => $auth,
            'sanitize_callback' => [__CLASS__, 'sanitize_text_3'],
        ]);

        // post (and any REST-enabled post type using the block editor)
        // Note: If meta is not registered for the current post type, WordPress will drop it on save.
        // This is why update prompts could "disappear" after reopening for CPTs/pages.
        $post_types = get_post_types(['show_in_rest' => true], 'names');
        if (!is_array($post_types) || empty($post_types)) {
            $post_types = ['post'];
        }
        foreach ($post_types as $pt) {
            if ($pt === self::TEMPLATE_CPT) {
                continue;
            }
            if ($pt === 'attachment') {
                continue;
            }

            register_post_meta($pt, self::META_MASTER, [
                'type'              => 'string',
                'single'            => true,
                'default'           => '',
                'show_in_rest'      => $rest_schema,
                'auth_callback'     => $auth,
                'sanitize_callback' => [__CLASS__, 'sanitize_textarea_3'],
            ]);

            register_post_meta($pt, self::META_TOPIC, [
                'type'              => 'string',
                'single'            => true,
                'default'           => '',
                'show_in_rest'      => $rest_schema,
                'auth_callback'     => $auth,
                'sanitize_callback' => [__CLASS__, 'sanitize_text_3'],
            ]);

            // Trial版：更新生成は提供しない（更新プロンプト/履歴は登録しない）

            register_post_meta($pt, 'ead_ext_json', [
                'type'          => 'string',
                'single'        => true,
                'default'       => '',
                'show_in_rest'  => $rest_schema,
                'auth_callback' => $auth,
            ]);
        }

        // Trial版：更新系の汎用register_metaも不要

    }

    /**
     * When the editor saves, apply queued ext values (stored in meta ead_ext_json) to real custom fields.
     * ead_ext_json is cleared after a successful apply.
     */
    public static function maybe_apply_ext_json($post_id, $post, $update) {
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        $raw = get_post_meta($post_id, 'ead_ext_json', true);
        if (!is_string($raw) || $raw === '') {
            return;
        }
        $data = json_decode($raw, true);
        if (!is_array($data)) {
            return;
        }
        foreach ($data as $k => $v) {
            if (!is_string($k) || $k === '') {
                continue;
            }
            // Allow flexible keys but keep them safe.
            if (!preg_match('/^[a-zA-Z0-9_\-]{1,64}$/', $k)) {
                continue;
            }
            if (is_array($v) || is_object($v)) {
                $v = wp_json_encode($v);
            }
            if (!is_string($v)) {
                $v = strval($v);
            }
            update_post_meta($post_id, $k, $v);
        }
        delete_post_meta($post_id, 'ead_ext_json');
    }


    public static function enqueue_editor_assets() {
        if ( ! function_exists('get_current_screen') ) { return; }
        $screen = get_current_screen();
        if ( ! $screen ) { return; }

        // IMPORTANT: Use per-asset filemtime so JS/CSS updates are not stuck by browser cache.
        // Using only __FILE__ caused editor-post.js changes to be ignored.
        $ver = defined('EAD_VERSION') ? EAD_VERSION : null;

        $tpl_js_ver   = $ver ?: @filemtime( plugin_dir_path(__FILE__) . 'assets/editor-template.js' );

        // Determine current post type.
        $post_type = '';
        if ( ! empty($screen->post_type) ) {
            $post_type = $screen->post_type;
        } elseif ( ! empty($_GET['post_type']) ) {
            $post_type = sanitize_key($_GET['post_type']);
        } elseif ( ! empty($_GET['post']) ) {
            $post_type = get_post_type( (int) $_GET['post'] );
        } else {
            $post_type = get_post_type();
        }

        $localized = [
            'ajax_url'  => admin_url('admin-ajax.php'),
            'nonce'     => wp_create_nonce('wp_rest'),
            // Dedicated nonce for admin-ajax draft meta persistence (REST may be disabled in some environments).
            'ajax_nonce'=> wp_create_nonce('ead_ajax'),
            'post_type' => $post_type,
            // Trial版：テンプレ生成/下書き生成のみ
            'restUrl'   => rest_url('ead/v1/generate_template'),
            // Admin edit URL base for redirect after creating drafts from templates
            'adminEditUrlBase' => admin_url('post.php?post='),
        ];

        // Enqueue only the relevant editor script for the current post type.
        $deps = [
            'wp-plugins',
            'wp-edit-post',
            'wp-editor',
            'wp-block-editor',
            'wp-blocks',
            'wp-rich-text',
            'wp-components',
            'wp-data',
            'wp-api-fetch',
            'wp-i18n',
            'wp-hooks',
            'wp-compose',
            'wp-element',
            'wp-url',
        ];

        if ( $post_type === 'ead_template' ) {
            wp_enqueue_script(
                'ead-template',
                plugins_url('assets/editor-template.js', __FILE__),
                $deps,
                $tpl_js_ver,
                true
            );
            wp_localize_script('ead-template', 'EAD', $localized);
        }
    }

    /**
     * Trial版：テンプレ合計が上限(5件)に達したら、一覧の「新規追加」ボタンを非表示
     * ※サーバ側で完全ブロックはしない（最小改修）。
     */
    public static function maybe_hide_add_new_template_button() {
        if (!is_admin()) return;
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || $screen->post_type !== self::TEMPLATE_CPT || $screen->base !== 'edit') return;

        $count = wp_count_posts(self::TEMPLATE_CPT);
        $total = 0;
        foreach (['publish','draft','pending','private','future'] as $st) {
            $total += isset($count->$st) ? (int)$count->$st : 0;
        }
        if ($total < 5) return;

        echo '<style>.page-title-action{display:none!important}</style>';
    }
}

Easy_AI_Director::init();

