<?php
if ( ! defined('ABSPATH') ) { exit; }

/**
 * Extract JSON object from raw model output.
 * Accepts either pure JSON or text containing a JSON object.
 *
 * @param string $raw
 * @return array|WP_Error
 */
function ead_extract_json( $raw ) {
    $raw = is_string($raw) ? trim($raw) : '';
    if ($raw === '') {
        return new WP_Error('ead_empty_response', 'AI response was empty.');
    }

    $data = json_decode($raw, true);
    if (is_array($data)) {
        return $data;
    }

    // Try to locate the first JSON object in the text.
    if (preg_match('/\{(?:[^{}]|(?R))*\}/s', $raw, $m)) {
        $candidate = $m[0];
        $data2 = json_decode($candidate, true);
        if (is_array($data2)) {
            return $data2;
        }
    }

    return new WP_Error('ead_invalid_json', 'AI response was not valid JSON.', array('raw' => $raw));
}

/**
 * Call AI provider (currently Gemini only).
 *
 * @param string $system_prompt
 * @param string $user_prompt
 * @param string $ai_provider
 * @param string $model
 * @param int    $creativity 0-100
 * @return string|WP_Error raw text
 */
function ead_call_ai( $system_prompt, $user_prompt, $ai_provider, $model, $creativity = 40 ) {
    $ai_provider = $ai_provider ? strtolower($ai_provider) : 'gemini';

    $settings = get_option('ead_ai_settings', array());
    $api_key = '';
    if (is_array($settings) && isset($settings['gemini_api_key'])) {
        $api_key = trim((string)$settings['gemini_api_key']);
    }
    if ($ai_provider !== 'gemini') {
        // Fallback to gemini for now (single provider)
        $ai_provider = 'gemini';
    }
    if ($api_key === '') {
        return new WP_Error('ead_missing_api_key', 'Gemini API key is not set.');
    }
    $model = $model ? $model : (isset($settings['default_model']) ? $settings['default_model'] : 'gemini-2.5-flash');

    // Convert creativity (0-100) to temperature (0-2).
    $temperature = max(0.0, min(2.0, (float)$creativity / 50.0));

    $endpoint = sprintf('https://generativelanguage.googleapis.com/v1beta/models/%s:generateContent?key=%s', rawurlencode($model), rawurlencode($api_key));

    $payload = array(
        'system_instruction' => array(
            'parts' => array(
                array('text' => (string)$system_prompt),
            ),
        ),
        'contents' => array(
            array(
                'role'  => 'user',
                'parts' => array(
                    array('text' => (string)$user_prompt),
                ),
            ),
        ),
        'generationConfig' => array(
            'temperature' => $temperature,
        ),
    );

    $resp = wp_remote_post($endpoint, array(
        'timeout' => 60,
        'headers' => array('Content-Type' => 'application/json'),
        'body'    => wp_json_encode($payload),
    ));

    if (is_wp_error($resp)) {
        return $resp;
    }
    $code = (int) wp_remote_retrieve_response_code($resp);
    $body = (string) wp_remote_retrieve_body($resp);
    if ($code < 200 || $code >= 300) {
        return new WP_Error('ead_ai_http_error', 'AI request failed.', array('status' => $code, 'body' => $body));
    }

    $json = json_decode($body, true);
    if (!is_array($json)) {
        return new WP_Error('ead_ai_bad_response', 'AI response was not JSON.', array('body' => $body));
    }

    $text = '';
    if (isset($json['candidates'][0]['content']['parts']) && is_array($json['candidates'][0]['content']['parts'])) {
        foreach ($json['candidates'][0]['content']['parts'] as $part) {
            if (isset($part['text'])) { $text .= (string)$part['text']; }
        }
        $text = trim($text);
    }
    if ($text === '') {
        // Some responses may use different fields; include raw for debugging.
        return new WP_Error('ead_ai_empty_text', 'AI response had no text.', array('json' => $json));
    }
    return $text;
}

/**
 * Apply slot updates to HTML.
 *
 * The editor stores placeholders like:
 *  - 【AI:1:】, 【AI:t_1_b2:ポイント追加】, etc.
 *  - 【AI_fix:title:】, 【AI_ext:faq:】 (when still present)
 *
 * This helper replaces placeholders for keys found in $updates.
 * The matching is intentionally permissive: it replaces any token whose
 * prefix is AI/AI_fix/AI_ext and whose key matches.
 */
if ( ! function_exists( 'ead_apply_updates_to_html' ) ) {
  
/**
 * Sanitize AI slot value so it can be safely inserted inside existing block wrapper tags.
 */
function ead_sanitize_slot_value($slot_key, $value) {
  $v = (string)$value;
  $v = trim($v);

  // Remove wrapping code fences if model accidentally included them
  $v = preg_replace('/^\s*```[a-zA-Z0-9_-]*\s*/', '', $v);
  $v = preg_replace('/\s*```\s*$/', '', $v);
  $v = trim($v);

  // p_* -> strip outer <p>...</p>
  if (preg_match('/^p_\d+$/', $slot_key)) {
    $v = preg_replace('/^\s*<p[^>]*>/i', '', $v);
    $v = preg_replace('/<\/p>\s*$/i', '', $v);
    return trim($v);
  }

  // h2_*, h3_*, h4_* -> strip outer heading wrapper if present
  if (preg_match('/^(h2|h3|h4)_\d+$/', $slot_key, $m)) {
    $tag = strtolower($m[1]);
    $v = preg_replace('/^\s*<'.$tag.'[^>]*>/i', '', $v);
    $v = preg_replace('/<\/'.$tag.'>\s*$/i', '', $v);
    return trim($v);
  }

  return $v;
}

function ead_apply_updates_to_html( $html, $updates ) {
    if ( ! is_string( $html ) || $html === '' ) {
      return is_string( $html ) ? $html : '';
    }
    if ( ! is_array( $updates ) || empty( $updates ) ) {
      return $html;
    }

    foreach ( $updates as $key => $value ) {
      $k = (string) $key;
      $v = is_scalar( $value ) ? (string) $value : wp_json_encode( $value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES );

      // Replace 【AI:<key>:...】
      $pattern_ai = '/【AI:' . preg_quote( $k, '/' ) . ':[^】]*】/u';
      $html = preg_replace( $pattern_ai, $v, $html );

      // Replace 【AI_fix:<key>:...】
      $pattern_fix = '/【AI_fix:' . preg_quote( $k, '/' ) . ':[^】]*】/u';
      $html = preg_replace( $pattern_fix, $v, $html );

      // Replace 【AI_ext:<key>:...】
      $pattern_ext = '/【AI_ext:' . preg_quote( $k, '/' ) . ':[^】]*】/u';
      $html = preg_replace( $pattern_ext, $v, $html );
    }

    return $html;
  }
}

// -----------------------------------------------------------------------------
// Inject empty AI placeholders (e.g. 【AI:p_2:】) into HTML for missing target slots.
// Used mainly in post_update where published posts typically don't keep AI markers.

/**
 * @param string $html
 * @param array  $target_keys e.g. ['h2_1','p_2']
 * @return string
 */
function ead_inject_ai_placeholders_for_targets( $html, $target_keys ) {
  if ( empty( $target_keys ) || ! is_string( $html ) || $html === '' ) {
    return $html;
  }

  // Fast skip: if all keys already exist, do nothing.
  $missing = [];
  foreach ( $target_keys as $k ) {
    $k = trim( (string) $k );
    if ( $k === '' ) { continue; }
    if ( preg_match( '/\[AI:' . preg_quote( $k, '/' ) . ':/u', $html ) ) {
      continue;
    }
    $missing[] = $k;
  }
  if ( empty( $missing ) ) {
    return $html;
  }

  // Group missing keys by tag prefix.
  // Supported prefixes: p, li, h1-h6.
  $by_tag = [];
  foreach ( $missing as $k ) {
    $prefix = preg_replace( '/_.*/', '', $k );
    $tag = null;
    if ( preg_match( '/^h[1-6]$/', $prefix ) ) {
      $tag = $prefix;
    } elseif ( $prefix === 'p' ) {
      $tag = 'p';
    } elseif ( $prefix === 'li' ) {
      $tag = 'li';
    }
    if ( ! $tag ) { continue; }
    $by_tag[ $tag ][] = $k;
  }
  if ( empty( $by_tag ) ) {
    return $html;
  }

  foreach ( $by_tag as $tag => $keys ) {
    // Determine which occurrence to target based on suffix number in key (e.g. p_2 => 2).
    // Apply in descending order so earlier replacements don't affect later counters.
    usort( $keys, function( $a, $b ) {
      $na = (int) preg_replace( '/^.*_/', '', $a );
      $nb = (int) preg_replace( '/^.*_/', '', $b );
      return $nb <=> $na;
    } );

    foreach ( $keys as $key ) {
      $n = (int) preg_replace( '/^.*_/', '', $key );
      if ( $n <= 0 ) { continue; }

      $counter = 0;
      $pattern = '/<\s*' . preg_quote( $tag, '/' ) . '(\s[^>]*)?>(.*?)<\s*\/\s*' . preg_quote( $tag, '/' ) . '\s*>/isu';
      $html = preg_replace_callback(
        $pattern,
        function( $m ) use ( $key, $n, &$counter, $tag ) {
          $counter++;
          if ( $counter !== $n ) {
            return $m[0];
          }
          // If this element already contains any AI marker, keep as-is.
          if ( preg_match( '/\[AI:' . preg_quote( $key, '/' ) . ':/u', $m[0] ) ) {
            return $m[0];
          }
          $attrs = isset( $m[1] ) ? $m[1] : '';
          $inner = isset( $m[2] ) ? $m[2] : '';
          // Append placeholder just before the closing tag, to keep existing markup intact.
          $inner .= '【AI:' . $key . ':】';
          return '<' . $tag . $attrs . '>' . $inner . '</' . $tag . '>';
        },
        $html,
        -1,
        $replaced_count
      );
    }
  }

  return $html;
}




// -----------------------------------------------------------------------------
// Slot-safe apply: keep the 【AI:key:...】 marker so future updates remain stable.
// Replaces the entire marker with a new marker that wraps the generated value.
// -----------------------------------------------------------------------------
if ( ! function_exists('ead_apply_slots_to_html') ) {
  function ead_apply_slots_to_html( $html, $slots ) {
    if ( ! is_string( $html ) || $html === '' ) {
      return is_string( $html ) ? $html : '';
    }
    if ( ! is_array( $slots ) || empty( $slots ) ) {
      return $html;
    }

    foreach ( $slots as $key => $value ) {
      $k = (string) $key;
      $v = is_scalar( $value ) ? (string) $value : wp_json_encode( $value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES );

      // Avoid breaking the marker
      $v = str_replace('】', '」', $v);

      $pattern_ai = '/【AI:' . preg_quote( $k, '/' ) . ':[^】]*】/u';
      $replacement = '【AI:' . $k . ':' . $v . '】';
      $html = preg_replace( $pattern_ai, $replacement, $html );
    }

    return $html;
  }
}
// -----------------------------------------------------------------------------
// Taxonomy helpers (category / tag)
// -----------------------------------------------------------------------------

if ( ! function_exists('ead_category_name_to_id') ) {
  /**
   * Resolve a category name to term_id. Creates the term if it doesn't exist.
   *
   * @param string $name
   * @return int term_id or 0
   */
  function ead_category_name_to_id( $name ) {
    $name = trim( (string) $name );
    if ( $name === '' ) return 0;

    $exists = term_exists( $name, 'category' );
    if ( is_array( $exists ) && ! empty( $exists['term_id'] ) ) return (int) $exists['term_id'];
    if ( is_int( $exists ) ) return (int) $exists;

    $created = wp_insert_term( $name, 'category' );
    if ( is_wp_error( $created ) ) {
      // If it already exists in a race, try resolving again.
      $exists2 = term_exists( $name, 'category' );
      if ( is_array( $exists2 ) && ! empty( $exists2['term_id'] ) ) return (int) $exists2['term_id'];
      if ( is_int( $exists2 ) ) return (int) $exists2;
      return 0;
    }
    return (int) $created['term_id'];
  }
}

if ( ! function_exists('ead_tag_name_to_id') ) {
  /**
   * Resolve a tag name to term_id. Creates the term if it doesn't exist.
   *
   * @param string $name
   * @return int term_id or 0
   */
  function ead_tag_name_to_id( $name ) {
    $name = trim( (string) $name );
    if ( $name === '' ) return 0;

    $exists = term_exists( $name, 'post_tag' );
    if ( is_array( $exists ) && ! empty( $exists['term_id'] ) ) return (int) $exists['term_id'];
    if ( is_int( $exists ) ) return (int) $exists;

    $created = wp_insert_term( $name, 'post_tag' );
    if ( is_wp_error( $created ) ) {
      $exists2 = term_exists( $name, 'post_tag' );
      if ( is_array( $exists2 ) && ! empty( $exists2['term_id'] ) ) return (int) $exists2['term_id'];
      if ( is_int( $exists2 ) ) return (int) $exists2;
      return 0;
    }
    return (int) $created['term_id'];
  }
}
