<?php
if ( ! defined('ABSPATH') ) exit;

require_once __DIR__ . '/rest-generate-common.php';

function ead_collect_ext_keys_from_text($text) {
  $keys = [];
  if (!is_string($text) || $text === '') return $keys;
  if (preg_match_all('/【AI_ext:([a-zA-Z0-9_\-]+):.*?】/u', $text, $m)) {
    foreach ($m[1] as $k) { $keys[$k] = true; }
  }
  return $keys;
}
function ead_collect_fix_types_from_text($text) {
  $types = [];
  if (!is_string($text) || $text === '') return $types;
  if (preg_match_all('/【AI_fix:([a-zA-Z0-9_\-]+):.*?】/u', $text, $m)) {
    foreach ($m[1] as $k) { $types[$k] = true; }
  }
  return $types;
}


/**
 * REST
 * - GET  /ead/v1/settings
 * - POST /ead/v1/generate
 *
 * Template editor expects: { success: true, blocks: [{clientId, primary, value}], edit_link? }
 * Post editor expects:      { ok: true, slots: {...} }
 *
 * This endpoint supports both request payload styles:
 * - legacy(post): { master, topic, targets, contextHtml, ... }
 * - template:     { post_id, mode:'template', master_instruction, topic, context_html, targets_json, ... }
 */

if ( ! function_exists('ead_get_ai_settings') ) {
  function ead_get_ai_settings() {
    $opt = get_option('ead_ai_settings', []);
    if (!is_array($opt)) $opt = [];
    $defaults = [
      'gemini_api_key' => '',
      'default_ai'     => 'gemini',
      'gemini_model'   => 'gemini-2.5-flash',
      'creativity'     => 40,
    ];
    return array_merge($defaults, $opt);
  }
}

add_action('rest_api_init', function () {

  register_rest_route('ead/v1', '/settings', [
    'methods' => 'GET',
    'permission_callback' => function () {
      return current_user_can('edit_posts');
    },
    'callback' => function () {
      $s = ead_get_ai_settings();
      return new WP_REST_Response([
        'ok' => true,
        'defaults' => [
          'ai_provider' => $s['default_ai'] ?: 'gemini',
          'ai_model'    => $s['gemini_model'] ?: 'gemini-2.5-flash',
          'creativity'  => intval($s['creativity']),
        ],
      ], 200);
    },
  ]);

  register_rest_route('ead/v1', '/generate_template', [
    'methods' => 'POST',
    'permission_callback' => function () {
      return current_user_can('edit_posts');
    },
    'callback' => function (WP_REST_Request $req) {
      $params = $req->get_json_params();
      if (!is_array($params)) $params = [];

      // Detect caller mode
      $mode = isset($params['mode']) ? sanitize_text_field($params['mode']) : '';

      // Trial版：更新生成は提供しない
      if (in_array($mode, ['post', 'update', 'post_update'], true)) {
        return new WP_REST_Response([
          'ok' => false,
          'message' => 'Trial版では更新生成は利用できません。',
        ], 400);
      }
      // IMPORTANT:
      // - If mode is explicitly provided, it must determine the behavior.
      // - Do NOT infer template mode merely by the presence of targets_json,
      //   because post update requests also include targets_json.
      if ($mode !== '') {
        $is_template = ($mode === 'template');
      } else {
        // Backward compatibility (older clients may not send mode)
        $is_template = isset($params['master_instruction']) && !isset($params['post_id']) && !isset($params['postId']);
      }

      // Read inputs (support both styles)
      $master = '';
      if (isset($params['master_instruction'])) $master = (string)$params['master_instruction'];
      elseif (isset($params['master'])) $master = (string)$params['master'];

      $topic = isset($params['topic']) ? (string)$params['topic'] : '';
      // Collect ext/fix requests from master/topic in template mode
      $apply_ext_keys = array_merge(
        ead_collect_ext_keys_from_text($master),
        ead_collect_ext_keys_from_text($topic)
      );
      $apply_fix_types_map = array_merge(
        ead_collect_fix_types_from_text($master),
        ead_collect_fix_types_from_text($topic)
      );
      $update_prompt = '';
      if (isset($params['update_prompt'])) $update_prompt = (string)$params['update_prompt'];
      elseif (isset($params['updatePrompt'])) $update_prompt = (string)$params['updatePrompt'];

      $post_id = isset($params['post_id']) ? intval($params['post_id']) : 0;
      if (!$post_id && isset($params['postId'])) $post_id = intval($params['postId']);

      $is_post_update = ($mode === 'post' || $mode === 'update' || $mode === 'post_update') && ($post_id > 0);

      $contextHtml = '';
      // For post update: if master prompt not provided, use the one stored in update history (initial generation)
      if ($is_post_update && trim($master) === '') {
        $hist = (string) get_post_meta($post_id, 'ead_update_history', true);
        if ($hist !== '') {
          // take last occurrence of master prompt block
          if (preg_match_all('/--- マスタープロンプト（全文） ---\n(.*?)\n------------------------------/us', $hist, $mm) && !empty($mm[1])) {
            $master = trim((string) end($mm[1]));
          }
        }
      }

      if (isset($params['context_html'])) {
        $contextHtml = (string)$params['context_html'];
      } elseif (isset($params['contextHtml'])) {
        $contextHtml = (string)$params['contextHtml'];
      }
      if ($topic !== '') {
        $contextHtml = str_replace('{{topic}}', $topic, $contextHtml);
      }

      $targets = [];
      if (isset($params['targets']) && is_array($params['targets'])) {
        $targets = $params['targets'];
      } elseif (isset($params['targets_json'])) {
        $decoded = json_decode((string)$params['targets_json'], true);
        if (is_array($decoded)) $targets = $decoded;
      }

      // Fallback: extract slot keys directly from HTML (supports user-defined keys)
      if (!is_array($targets)) $targets = [];
      $found = [];
      if ($contextHtml !== '') {
        if (preg_match_all('/【AI:([^:】]+):([^】]*)】/u', $contextHtml, $m)) {
          if (isset($m[1]) && is_array($m[1])) {
            foreach ($m[1] as $k) {
              $k = trim((string)$k);
              if ($k !== '') $found[] = $k;
            }
          }
        }
      }
      if ($found) {
        $targets = array_values(array_unique(array_merge($targets, $found)));
      }

      // AI settings
      $s = ead_get_ai_settings();
      $ai_provider = isset($params['ai_provider']) ? sanitize_text_field($params['ai_provider']) : ($s['default_ai'] ?: 'gemini');
      $ai_model    = isset($params['ai_model']) ? sanitize_text_field($params['ai_model']) : ($s['gemini_model'] ?: 'gemini-2.5-flash');
      $creativity  = isset($params['creativity']) ? intval($params['creativity']) : intval($s['creativity']);
      if ($creativity < 0) $creativity = 0;
      if ($creativity > 100) $creativity = 100;

      if ($ai_provider !== 'gemini') $ai_provider = 'gemini';

      $api_key = isset($s['gemini_api_key']) ? trim((string)$s['gemini_api_key']) : '';
      if ($api_key === '') {
        $err = 'Gemini APIキーが未設定です（Easy AI Director → AI設定）。';
        return new WP_REST_Response($is_template ? ['success'=>false,'message'=>$err] : ['ok'=>false,'error'=>$err], 400);
      }

            // Provide existing categories/tags as choices for AI (categories: all, tags: top 200 by count)
      $ead_categories = [];
      $ead_tags = [];

      $terms_cat = get_terms([
        'taxonomy'   => 'category',
        'hide_empty' => false,
      ]);
      if (!is_wp_error($terms_cat) && is_array($terms_cat)) {
        foreach ($terms_cat as $t) {
          if (!is_object($t)) continue;
          $ead_categories[] = [
            'name' => (string)$t->name,
            'slug' => (string)$t->slug,
          ];
        }
      }

      $terms_tag = get_terms([
        'taxonomy'   => 'post_tag',
        'hide_empty' => false,
        'orderby'    => 'count',
        'order'      => 'DESC',
        'number'     => 200,
      ]);
      if (!is_wp_error($terms_tag) && is_array($terms_tag)) {
        foreach ($terms_tag as $t) {
          if (!is_object($t)) continue;
          $ead_tags[] = [
            'name'  => (string)$t->name,
            'slug'  => (string)$t->slug,
            'count' => isset($t->count) ? intval($t->count) : 0,
          ];
        }
      }

// Prompt: emphasize context-aware generation based on full structure
      $system  = "あなたはWordPressのブロックエディタで記事を編集するプロの編集者です。\n";
      $system .= "必ずJSONのみで返答してください。前置き・説明・Markdownは禁止。\n";

      // ---------------------------------------------------------
      // Detect which AI_fix / AI_ext slots are enabled by master prompt.
      // Rule:
      // - If a slot is NOT present in master prompt, we must NOT generate/apply it.
      // - AI_fix slots may have empty instruction and still be generated internally.
      // - AI_ext slots require an instruction (text between ':' and '】').
      // ---------------------------------------------------------
      $apply_fix_types = [];
      if (is_string($master) && $master !== '' && preg_match_all('/【AI_fix:(title|slug|category|tag):/u', $master, $mfix)) {
        $apply_fix_types = array_values(array_unique($mfix[1]));
      }

      $apply_ext_keys = [];
      if (is_string($master) && $master !== '' && preg_match_all('/【AI_ext:([a-zA-Z0-9_\-]+):([^】]+)】/u', $master, $mext, PREG_SET_ORDER)) {
        foreach ($mext as $mm) {
          $k = trim((string)$mm[1]);
          $instr = trim((string)$mm[2]);
          if ($k !== '' && $instr !== '') {
            $apply_ext_keys[$k] = true;
          }
        }
      }

      // Internal default rules (applied only when no explicit instruction exists in slot/master)
      $system .= "【優先順位】
";
      $system .= "1) 各スロット内の指示（:】の間）
";
      $system .= "2) マスタープロンプト
";
      $system .= "3) 内部デフォルト規約（本項）
";
      $system .= "矛盾する場合は必ず上位を優先する。

";
      $system .= "【AI_fix デフォルト】
";
      $system .= "- title: 指示が無い場合は40字前後、{{topic}}を含める。過度な煽りは禁止。
";
      $system .= "- slug: 指示が無い場合は20字未満、基本は{{topic}}を含める。長すぎる場合は一般的略称に置換して短縮。英語/ローマ字、a-z0-9- のみ。
";
      $system .= "- category: 既存カテゴリ一覧から選ぶ。指示が無い場合は1〜3個。新規作成は指示がある場合のみ可。複合語は分解。
";
      $system .= "- tag: 既存タグ一覧から選ぶ。指示が無い場合は最大5個。指示があれば増やしてよい。新規作成は指示がある場合のみ可。複合語は分解。

";
      $system .= "【AI_ext デフォルト】
";
      $system .= "- 【AI_ext:key:指示】はkeyが任意。ただし『マスタープロンプト内に存在するキー』かつ『指示があるキー』のみ生成する。勝手に新しいキーを作らない。
";
      $system .= "- extオブジェクトには要求キーのみ入れる。要求キーが無い場合は ext を空オブジェクト {} にする。

";
      $system .= "【updates デフォルト】
";
      $system .= "- targetsに含まれるキーは必ず全てupdatesに出力（欠け禁止）。
";
      $system .= "- 指示が無い場合でも本文全体の文脈に合わせて自然な文章で埋める。

";
      $system .= "【出力フォーマット】
";
      $system .= "- JSONのみ。updates/fix/extの3キーを含める（fix/extは空オブジェクト {} でも可）。
";
      $system .= "- updatesはオブジェクト（数値キーは\"1\"のように文字列キーで返す）。targetsに含まれるキーは必ず全て含める（欠け禁止）。
";
      $system .= "- fix/extは『要求されたスロット』のみ出力し、勝手に追加しない。fixのcategory/tagは配列で返す。

";


      $user  = "【目的】\n";
      $user .= "テンプレート/記事本文の全体構成（見出し階層・前後の段落・表/リストの内容）を読み取り、文脈に合う最適な文章を生成してください。\n";
      $user .= "各スロットは単体で書くのではなく、全体の流れが自然になるようにトーンと情報量を揃えてください。\n\n";
      $user .= "【マスタープロンプト】\n" . $master . "\n\n";
      $user .= "【トピック】\n" . $topic . "\n\n";
      $user .= "【既存カテゴリ一覧】\n" . json_encode($ead_categories, JSON_UNESCAPED_UNICODE) . "\n\n";
      $user .= "【既存タグ一覧（上位200）】\n" . json_encode($ead_tags, JSON_UNESCAPED_UNICODE) . "\n\n";
      $user .= "【本文（現在のHTML/構造を含む）】\n" . $contextHtml . "\n\n";
      $user .= "【更新対象（targets）】\n" . json_encode($targets, JSON_UNESCAPED_UNICODE) . "\n\n";

      // Output rules: keep these internal so users don't have to paste long JSON specs

      if (!function_exists('ead_find_nested_key')) {
        function ead_find_nested_key($data, $key) {
          if (is_array($data)) {
            if (isset($data[$key]) && is_array($data[$key])) return $data[$key];
            foreach ($data as $v) {
              $found = ead_find_nested_key($v, $key);
              if (is_array($found)) return $found;
            }
          }
          return null;
        }
      }


      // Ensure all targets are filled: retry once for missing keys
      $targets_keys = [];
      foreach ($targets as $t) {
        if (is_string($t) && trim($t) !== '') $targets_keys[] = trim($t);
      }

      $updates_candidate = null;
      if (isset($obj['updates']) && is_array($obj['updates'])) {
        $updates_candidate = $obj['updates'];
      } elseif (isset($obj['blocks']) && is_array($obj['blocks'])) {
        // normalize blocks -> updates
        $updates_candidate = [];
        foreach ($obj['blocks'] as $b) {
          $pk = isset($b['primary']) ? (string)$b['primary'] : '';
          if ($pk === '') continue;
          $updates_candidate[$pk] = isset($b['value']) ? (string)$b['value'] : '';
        }
        $obj['updates'] = $updates_candidate;
      }

      if (is_array($updates_candidate)) {
        $missing = [];
        foreach ($targets_keys as $k) {
          if (!array_key_exists($k, $updates_candidate)) $missing[] = $k;
        }

        if (!empty($missing)) {
          $user_retry = "【追加依頼】\\n";
          $user_retry .= "以下のキーが slots から欠けています。必ず全て埋めてください。\\n";
          $user_retry .= json_encode($missing, JSON_UNESCAPED_UNICODE) . "\\n\\n";
          $user_retry .= "出力は JSON のみ。形式は { \"updates\": { \"KEY\": \"VALUE\" } } です。\\n";

          $payload_retry = [
            'contents' => [
              [
                'role' => 'user',
                'parts' => [
                  ['text' => $system . "\\n" . $user . "\\n" . $user_retry]
                ],
              ],
            ],
            'generationConfig' => [
              'temperature' => $temperature,
            ],
          ];

          $resp2 = wp_remote_post($endpoint, [
            'timeout' => 60,
            'headers' => [
              'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($payload_retry, JSON_UNESCAPED_UNICODE),
          ]);

          if (!is_wp_error($resp2)) {
            $code2 = wp_remote_retrieve_response_code($resp2);
            $body2 = wp_remote_retrieve_body($resp2);
            if ($code2 >= 200 && $code2 < 300) {
              $json2 = json_decode($body2, true);
              $text2 = '';
              if (is_array($json2) && isset($json2['candidates'][0]['content']['parts'][0]['text'])) {
                $text2 = (string)$json2['candidates'][0]['content']['parts'][0]['text'];
              } elseif (is_array($json2) && isset($json2['candidates'][0]['content']['parts'])) {
                $parts2 = $json2['candidates'][0]['content']['parts'];
                if (is_array($parts2)) {
                  foreach ($parts2 as $p) {
                    if (isset($p['text'])) $text2 .= (string)$p['text'];
                  }
                }
              }
              if (trim($text2) !== '') {
                $cand2 = trim($text2);
                $cand2 = preg_replace('/```(?:json)?/i', '', $cand2);
                $cand2 = str_replace('```', '', $cand2);
                $cand2 = trim($cand2);
                $obj2 = json_decode($cand2, true);
                if (!is_array($obj2)) {
                  if (preg_match('/\{[\s\S]*\}/u', $cand2, $m3)) {
                    $obj2 = json_decode($m3[0], true);
                  }
                }
                if (is_array($obj2) && isset($obj2['updates']) && is_array($obj2['updates'])) {
                  foreach ($obj2['updates'] as $mk => $mv) {
                    $mk = is_string($mk) ? trim($mk) : (string)$mk;
                    if ($mk === '') continue;
                    $obj['updates'][$mk] = (string)$mv;
                  }
                }
              }
            }
          }
        }
      }
      if ($is_template) {
        $user .= "【出力ルール（テンプレート下書き生成）】\n";
        $user .= "- 必ずJSONのみで返答する。\n";
        $user .= "- blocks というキーは使わない。必ず updates/fix/ext を使う。\n";
        $user .= "- slots には targets に含まれるキーを必ず全て含める（欠け禁止）。\n";
        $user .= "- fix はマスタープロンプトに存在するAI_fixスロットのみ出力（存在しないものは出力しない / 変更しない）。\n";
        $user .= "- ext はマスタープロンプトに存在し、かつ指示があるAI_extキーのみ出力（存在しないものは出力しない / 追加しない）。\n";
        $user .= "- fix/ext が要求されていない場合は空オブジェクト {} を出力する。\n";

        $user .= "\n【要求されたAI_fix】\n" . json_encode(array_values($apply_fix_types), JSON_UNESCAPED_UNICODE) . "\n";
        $user .= "【要求されたAI_ext】\n" . json_encode(array_keys($apply_ext_keys), JSON_UNESCAPED_UNICODE) . "\n\n";

        $user .= "出力JSON例（fix/extが空でも可）:\n";
        $user .= "{ \"updates\": {\"p_1\":\"...\"}, \"fix\": {}, \"ext\": {} }\n\n";
      } else {
        $user .= "【出力ルール（投稿更新生成）】\n";
        $user .= "- 必ずJSONのみで返答する。\n";
        $user .= "- slots には更新対象のスロットキーと本文差し替え用の値を入れる。\n\n";
      }

      // Call Gemini
      $endpoint = 'https://generativelanguage.googleapis.com/v1beta/models/' . rawurlencode($ai_model) . ':generateContent?key=' . rawurlencode($api_key);

      $temperature = max(0.0, min(1.0, floatval($creativity) / 100.0));

      $payload = [
        'contents' => [
          [
            'role' => 'user',
            'parts' => [
              ['text' => $system . "\n" . $user]
            ],
          ],
        ],
        'generationConfig' => [
          'temperature' => $temperature,
        ],
      ];


      $resp = wp_remote_post($endpoint, [
        'timeout' => 60,
        'headers' => [
          'Content-Type' => 'application/json',
        ],
        'body' => wp_json_encode($payload, JSON_UNESCAPED_UNICODE),
      ]);

      if (is_wp_error($resp)) {
        $err = $resp->get_error_message();
        return new WP_REST_Response($is_template ? ['success'=>false,'message'=>$err] : ['ok'=>false,'error'=>$err], 500);
      }

      $code = wp_remote_retrieve_response_code($resp);
      $body = wp_remote_retrieve_body($resp);

      if ($code < 200 || $code >= 300) {
        $msg = 'Gemini API error (' . $code . ')';
        return new WP_REST_Response($is_template ? ['success'=>false,'message'=>$msg,'debug'=>['body'=>$body]] : ['ok'=>false,'error'=>$msg,'debug'=>['body'=>$body]], 500);
      }

      $json = json_decode($body, true);
      $text = '';
      if (is_array($json) && isset($json['candidates'][0]['content']['parts'][0]['text'])) {
        $text = (string)$json['candidates'][0]['content']['parts'][0]['text'];
      } elseif (is_array($json) && isset($json['candidates'][0]['content']['parts'])) {
        // join parts if multiple
        $parts = $json['candidates'][0]['content']['parts'];
        if (is_array($parts)) {
          foreach ($parts as $p) {
            if (isset($p['text'])) $text .= (string)$p['text'];
          }
        }
      }

      if (trim($text) === '') {
        $msg = 'Geminiの応答テキストが空でした。';
        return new WP_REST_Response($is_template ? ['success'=>false,'message'=>$msg,'debug'=>['body'=>$body]] : ['ok'=>false,'error'=>$msg,'debug'=>['body'=>$body]], 500);
      }

      // Extract JSON object from text (remove code fences)
      $candidate = trim($text);
      $candidate = preg_replace('/```(?:json)?/i', '', $candidate);
      $candidate = str_replace('```', '', $candidate);
      $candidate = trim($candidate);

      // Try direct decode first
      $obj = json_decode($candidate, true);

      // If decode fails, try to extract first {...} block
      if (!is_array($obj)) {
        if (preg_match('/\{[\s\S]*\}/u', $candidate, $m2)) {
          $obj = json_decode($m2[0], true);
        }
      }

      if (!is_array($obj)) {
        $msg = 'AIの出力がJSONとして解析できませんでした。';
        return new WP_REST_Response($is_template ? ['success'=>false,'message'=>$msg,'debug'=>['raw'=>$text]] : ['ok'=>false,'error'=>$msg,'debug'=>['raw'=>$text]], 500);
      }


      if ($is_template) {
        // Expect slots keyed by slot key + optional fix/ext outputs
        $updates = (isset($obj['updates']) && is_array($obj['updates'])) ? $obj['updates'] : ead_find_nested_key($obj, 'updates');
        if (!$updates) {
          // Fallback: some models may return {blocks:[{primary,value}...]} instead of updates
          if (isset($obj['blocks']) && is_array($obj['blocks'])) {
            $updates = [];
            foreach ($obj['blocks'] as $b) {
              $pk = isset($b['primary']) ? (string)$b['primary'] : '';
              if ($pk === '') continue;
              $updates[$pk] = isset($b['value']) ? (string)$b['value'] : '';
            }
          }
        }
        if (!$updates) {
          if (!empty($targets)) {
            $err = 'updates が見つかりませんでした。';
            return new WP_REST_Response(['success'=>false,'message'=>$err,'debug'=>['data'=>$obj,'raw'=>$text]], 500);
          }
          $updates = [];
        }

        $fix_raw = (isset($obj['fix']) && is_array($obj['fix'])) ? $obj['fix'] : (ead_find_nested_key($obj, 'fix') ?: []);
        $ext_raw = (isset($obj['ext']) && is_array($obj['ext'])) ? $obj['ext'] : (ead_find_nested_key($obj, 'ext') ?: []);

        // -------------------------------------------------------
        // Strictly filter fix/ext by slots declared in master prompt.
        // If a slot is not present, ignore even if the model returned it.
        // -------------------------------------------------------
        $fix = [];
        if (is_array($fix_raw) && !empty($apply_fix_types)) {
          foreach ($apply_fix_types as $ft) {
            if (is_string($ft) && $ft !== '' && array_key_exists($ft, $fix_raw)) {
              $fix[$ft] = $fix_raw[$ft];
            }
          }
        }

        $ext = [];
        if (is_array($ext_raw) && !empty($apply_ext_keys)) {
          foreach (array_keys($apply_ext_keys) as $ek) {
            if (is_string($ek) && $ek !== '' && array_key_exists($ek, $ext_raw)) {
              $ext[$ek] = $ext_raw[$ek];
            }
          }
        }

        // Build final HTML by replacing slot markers in the template HTML (global replace)
        
        // Normalize list-form slots (["...","..."]) into keyed slots using targets order
        if (is_array($updates) && array_keys($updates) === range(0, count($updates) - 1) && !empty($targets_keys)) {
          $mapped_updates = [];
          $n = min(count($targets_keys), count($updates));
          for ($i = 0; $i < $n; $i++) {
            $mapped_updates[(string)$targets_keys[$i]] = (string)$updates[$i];
          }
          $updates = $mapped_updates;
        }

$final_html = $contextHtml;
        if ($topic !== '') { $final_html = str_replace('{{topic}}', $topic, $final_html); }

        foreach ($updates as $slot_key => $val) {
          $slot_key = is_string($slot_key) ? trim($slot_key) : (string)$slot_key;
          if ($slot_key === '') continue;
          $needle_re = '/【AI:' . preg_quote($slot_key, '/') . ':[^】]*】/u';
	          // Replace any 【AI:slot_key:...】 marker with generated value
	          $final_html = preg_replace($needle_re, (string)$val, $final_html);
        }

        // Replace AI_fix placeholders in body (optional)
        foreach (['title','slug','category','tag'] as $k) {
          if (!isset($fix[$k])) continue;
          $vv = $fix[$k];
          if (is_array($vv)) { $vv = implode('、', array_map('strval', $vv)); }
          $final_html = str_replace('【AI_fix:' . $k . ':】', (string)$vv, $final_html);
        }

        // Replace AI_ext placeholders in body and collect meta
        $ext_meta = [];
        foreach ($ext as $k => $v) {
          if (!is_string($k)) continue;
          $k = trim($k);
          if ($k === '') continue;
          $v = (string)$v;
          $final_html = str_replace('【AI_ext:' . $k . ':】', $v, $final_html);
          $ext_meta[$k] = $v;
        }



        // If placeholders still remain, retry once for remaining keys (AI / AI_fix / AI_ext)
        $remaining_ai = [];
        $remaining_fix = [];
        $remaining_ext = [];

        // AI slots
        if (preg_match_all('/【AI:([^:】]+):([^】]*)】/u', $final_html, $m_ai)) {
          if (isset($m_ai[1]) && is_array($m_ai[1])) {
            foreach ($m_ai[1] as $rk) {
              $rk = trim((string)$rk);
              if ($rk !== '') $remaining_ai[] = $rk;
            }
          }
        }
        $remaining_ai = array_values(array_unique($remaining_ai));

        // Fixed fields (title/slug/category/tag) - only when declared in master prompt
        $fix_candidates = !empty($apply_fix_types) ? array_values($apply_fix_types) : [];
        foreach ($fix_candidates as $fk) {
          if (strpos($final_html, '【AI_fix:' . $fk . ':】') !== false) {
            $remaining_fix[] = $fk;
          }
        }

        // Custom fields (any key) - only when declared in master prompt
        if (preg_match_all('/【AI_ext:([^:]+):】/u', $final_html, $m_ext)) {
          if (isset($m_ext[1]) && is_array($m_ext[1])) {
            foreach ($m_ext[1] as $ek) {
              $ek = trim((string)$ek);
              if ($ek !== '' && isset($apply_ext_keys[$ek])) $remaining_ext[] = $ek;
            }
          }
        }
        $remaining_ext = array_values(array_unique($remaining_ext));

        if (!empty($remaining_ai) || !empty($remaining_fix) || !empty($remaining_ext)) {
          $user_retry2 = "【追加依頼】\n";
          $user_retry2 .= "本文内に未置換のスロットが残っています。必ず全て埋めてください。\n\n";
          if (!empty($remaining_ai)) {
            $user_retry2 .= "updates に必要なキー:\n" . json_encode($remaining_ai, JSON_UNESCAPED_UNICODE) . "\n\n";
          }
          if (!empty($remaining_fix)) {
            $user_retry2 .= "fix に必要なキー:\n" . json_encode($remaining_fix, JSON_UNESCAPED_UNICODE) . "\n\n";
          }
          if (!empty($remaining_ext)) {
            $user_retry2 .= "ext に必要なキー:\n" . json_encode($remaining_ext, JSON_UNESCAPED_UNICODE) . "\n\n";
          }
          $user_retry2 .= "出力は JSON のみ。\n";
          $user_retry2 .= "形式:\n";
          $user_retry2 .= "{\n";
          $user_retry2 .= "  \"updates\": { \"KEY\": \"VALUE\" },\n";
          $user_retry2 .= "  \"fix\": { \"title\": \"...\", \"slug\": \"...\", \"category\": \"...\", \"tag\": \"...\" },\n";
          $user_retry2 .= "  \"ext\": { \"KEY\": \"VALUE\" }\n";
          $user_retry2 .= "}\n";

          $payload_retry2 = [
            'contents' => [
              [
                'role' => 'user',
                'parts' => [
                  ['text' => $system . "\n" . $user . "\n" . $user_retry2]
                ],
              ],
            ],
            'generationConfig' => [
              'temperature' => $temperature,
            ],
          ];

          $resp3 = wp_remote_post($endpoint, [
            'timeout' => 60,
            'headers' => [ 'Content-Type' => 'application/json' ],
            'body' => wp_json_encode($payload_retry2, JSON_UNESCAPED_UNICODE),
          ]);

          if (!is_wp_error($resp3)) {
            $code3 = wp_remote_retrieve_response_code($resp3);
            $body3 = wp_remote_retrieve_body($resp3);
            if ($code3 >= 200 && $code3 < 300) {
              $json3 = json_decode($body3, true);
              $text3 = '';
              if (is_array($json3) && isset($json3['candidates'][0]['content']['parts'][0]['text'])) {
                $text3 = (string)$json3['candidates'][0]['content']['parts'][0]['text'];
              } elseif (is_array($json3) && isset($json3['candidates'][0]['content']['parts'])) {
                $parts3 = $json3['candidates'][0]['content']['parts'];
                if (is_array($parts3)) foreach ($parts3 as $p) if (isset($p['text'])) $text3 .= (string)$p['text'];
              }
              if (trim($text3) !== '') {
                $cand3 = trim($text3);
                $cand3 = preg_replace('/```(?:json)?/i', '', $cand3);
                $cand3 = str_replace('```', '', $cand3);
                $cand3 = trim($cand3);
                $obj3 = json_decode($cand3, true);
                if (!is_array($obj3)) {
                  if (preg_match('/\{[\s\S]*\}/u', $cand3, $m4)) $obj3 = json_decode($m4[0], true);
                }

                // Apply slots (supports associative or list arrays)
                if (is_array($obj3) && isset($obj3['updates']) && is_array($obj3['updates'])) {
                  $final_html = ead_apply_updates_to_html($final_html, $obj3['updates'], $remaining_ai);
                }

                // Apply fix
                if (is_array($obj3) && isset($obj3['fix']) && is_array($obj3['fix'])) {
                  foreach ($obj3['fix'] as $fk => $fv) {
                    $fk = is_string($fk) ? trim($fk) : '';
	                    if ($fk === '' || empty($apply_fix_types) || !in_array($fk, $apply_fix_types, true)) continue;
                    $fv_text = $fv;
                    if (is_array($fv_text)) { $fv_text = implode('、', array_map('strval', $fv_text)); }
                    $final_html = str_replace('【AI_fix:' . $fk . ':】', (string)$fv_text, $final_html);
                    $fix_meta[$fk] = $fv;
                  }
                }

                // Apply ext
                if (is_array($obj3) && isset($obj3['ext']) && is_array($obj3['ext'])) {
                  foreach ($obj3['ext'] as $ek => $ev) {
                    $ek = is_string($ek) ? trim($ek) : '';
	                    if ($ek === '' || empty($apply_ext_keys) || !isset($apply_ext_keys[$ek])) continue;
                    $ev = (string)$ev;
                    $final_html = str_replace('【AI_ext:' . $ek . ':】', $ev, $final_html);
                    $ext_meta[$ek] = $ev;
                  }
                }
              }
            }
          }
        }

        // Determine draft title (prefer AI_fix:title > topic > template title)
        $template_id = isset($params['post_id']) ? intval($params['post_id']) : 0;
        $template_title = '';
        if ($template_id > 0) {
          $tp = get_post($template_id);
          if ($tp && !is_wp_error($tp)) $template_title = (string)$tp->post_title;
        }

        $title = '';
        if (isset($fix['title']) && is_string($fix['title'])) $title = trim((string)$fix['title']);
        if ($title === '') $title = trim((string)$topic);
        if ($title === '') $title = ($template_title !== '' ? $template_title : '下書き');

        // Create draft post
        $postarr = [
          'post_type'    => 'post',
          'post_status'  => 'draft',
          'post_title'   => $title,
          'post_content' => $final_html,
        ];

        // Apply slug if provided
        if (isset($fix['slug']) && is_string($fix['slug'])) {
          $slug = sanitize_title((string)$fix['slug']);
          if ($slug !== '') $postarr['post_name'] = $slug;
        }

        $new_post_id = wp_insert_post($postarr, true);

        if (is_wp_error($new_post_id) || $new_post_id <= 0) {
          $err = is_wp_error($new_post_id) ? $new_post_id->get_error_message() : '下書きの作成に失敗しました。';
          return new WP_REST_Response(['success'=>false,'message'=>$err], 500);
        }

        // Apply category/tag if provided (by name, create if missing)
        if (isset($fix['category']) && (is_string($fix['category']) || is_array($fix['category']))) {
          $cats_raw = is_array($fix['category']) ? $fix['category'] : preg_split('/[、,
]+/u', (string)$fix['category']);
$cat_ids = [];
          foreach ($cats_raw as $cn) {
            $cn = trim((string)$cn);
            if ($cn === '') continue;
            $term = term_exists($cn, 'category');
            if (!$term) $term = wp_insert_term($cn, 'category');
            if (!is_wp_error($term) && isset($term['term_id'])) $cat_ids[] = intval($term['term_id']);
          }
          if ($cat_ids) wp_set_post_categories($new_post_id, $cat_ids, false);
        }

        if (isset($fix['tag']) && (is_string($fix['tag']) || is_array($fix['tag']))) {
          $tags_raw = is_array($fix['tag']) ? $fix['tag'] : preg_split('/[、,
]+/u', (string)$fix['tag']);
$tags = [];
          foreach ($tags_raw as $tn) {
            $tn = trim((string)$tn);
            if ($tn === '') continue;
            $tags[] = $tn;
          }
          if ($tags) wp_set_post_tags($new_post_id, $tags, false);
        }

        // Save ext meta
        foreach ($ext_meta as $k => $v) {
          update_post_meta($new_post_id, $k, $v);
        }

        // Save generation metadata (initial generation)
        update_post_meta($new_post_id, 'ead_generated_at', current_time('mysql'));
        if ($template_id > 0) update_post_meta($new_post_id, 'ead_template_id', $template_id);
        if ($template_title !== '') update_post_meta($new_post_id, 'ead_template_title', $template_title);
        if ($topic !== '') update_post_meta($new_post_id, 'ead_topic', $topic);
        if ($master !== '') update_post_meta($new_post_id, 'ead_master_prompt', $master);

        // Update history (initial generation)
        $hist_prev = (string) get_post_meta($new_post_id, 'ead_update_history', true);
        $hist_prev = trim($hist_prev);

        $hist_entry  = "【初回生成】\n";
        $hist_entry .= "日時: " . current_time('mysql') . "\n";
        if ($template_title !== '' || $template_id > 0) {
          $hist_entry .= "テンプレート: " . ($template_title !== '' ? $template_title : '（無題）') . " (ID:" . intval($template_id) . ")\n";
        }
        if ($topic !== '') {
          $hist_entry .= "トピック: " . $topic . "\n";
        }
        if ($ai_provider !== '' || $ai_model !== '') {
          $hist_entry .= "AI: " . ($ai_provider !== '' ? $ai_provider : '-') . " / " . ($ai_model !== '' ? $ai_model : '-') . "\n";
        }
        $hist_entry .= "文章の自由度: " . (string) $creativity . "\n";
        $hist_entry .= "\n";
        $hist_entry .= "--- マスタープロンプト（全文） ---\n";
        $hist_entry .= ($master !== '' ? $master : '（未設定）') . "\n";
        $hist_entry .= "------------------------------\n";

        $hist_next = $hist_prev !== '' ? ($hist_prev . "\n\n" . $hist_entry) : $hist_entry;
        update_post_meta($new_post_id, 'ead_update_history', $hist_next);

        $edit_link = admin_url('post.php?post=' . intval($new_post_id) . '&action=edit');

        return new WP_REST_Response([
          'success' => true,
          'blocks' => [],
          'edit_link' => $edit_link,
        ], 200);
      }

      $updates = (isset($obj['updates']) && is_array($obj['updates'])) ? $obj['updates'] : ead_find_nested_key($obj, 'updates');
      // Allow empty slots only when there are no targets (e.g., user didn't specify any slots)
      if (!$updates) {
        if (!empty($targets)) {
          $err = 'updates が見つかりませんでした。';
          return new WP_REST_Response(['ok'=>false,'error'=>$err,'debug'=>['data'=>$obj]], 500);
        }
        $updates = [];
      }

      
      // Post update: apply to post content + optional fix/ext updates, then record history
      if ($is_post_update) {
        $post = get_post($post_id);
        if (!$post || $post->post_type === 'revision') {
          return new WP_REST_Response(['ok' => false, 'error' => 'invalid_post'], 400);
        }

        // Current content (editor state) is provided from UI; fall back to DB if missing.
        $current_html = is_string($context_html) && $context_html !== '' ? $context_html : (string) $post->post_content;

        // Apply generic slots (【AI:key:...】) and topic replacement.
        $slots_in = [];
        if (isset($obj['slots']) && is_array($obj['slots'])) { $slots_in = $obj['slots']; }
        else { $slots_in = (array)$updates; }
        $updated_html = ead_apply_updates_to_html($current_html, $slots_in);
if (is_string($topic) && $topic !== '') {
          $updated_html = str_replace(['{{topic}}', '｛｛topic｝｝'], $topic, $updated_html);
        }

        // Apply AI_fix only when explicitly instructed in update_prompt.
        $apply_fix = is_string($update_prompt) && preg_match('/【AI_fix:(title|slug|category|tag):/u', $update_prompt);

        $fix_out = [
          'apply' => (bool) $apply_fix,
          'title' => null,
          'slug' => null,
          'category_names' => [],
          'tag_names' => [],
          'category_ids' => [],
          'tag_ids' => [],
        ];

        if ($apply_fix && is_array($fix)) {
          if (!empty($fix['title'])) {
            $fix_out['title'] = (string) $fix['title'];
          }
          if (!empty($fix['slug'])) {
            $fix_out['slug'] = sanitize_title((string) $fix['slug']);
          }

          // category/tag are arrays of term names
          if (!empty($fix['category']) && is_array($fix['category'])) {
            $names = array_values(array_filter(array_map('strval', $fix['category'])));
            $fix_out['category_names'] = $names;
            foreach ($names as $nm) {
              $id = ead_category_name_to_id($nm);
              if ($id) $fix_out['category_ids'][] = (int) $id;
            }
          }
          if (!empty($fix['tag']) && is_array($fix['tag'])) {
            $names = array_values(array_filter(array_map('strval', $fix['tag'])));
            $fix_out['tag_names'] = $names;
            foreach ($names as $nm) {
              $id = ead_tag_name_to_id($nm);
              if ($id) $fix_out['tag_ids'][] = (int) $id;
            }
          }
        }

        // Apply AI_ext only for keys explicitly present in update_prompt.
        $ext_requested = [];
        if (is_string($update_prompt) && preg_match_all('/【AI_ext:([a-zA-Z0-9_\-]+):/u', $update_prompt, $mm)) {
          $req_keys = array_values(array_unique($mm[1]));
          if (is_array($ext)) {
            foreach ($req_keys as $k) {
              if (array_key_exists($k, $ext) && is_string($ext[$k])) {
                $ext_requested[$k] = $ext[$k];
              }
            }
          }
        }

        // Replace ext slots in HTML (even if meta is not persisted yet).
        if (!empty($ext_requested)) {
          foreach ($ext_requested as $k => $v) {
            $updated_html = preg_replace('/【AI_ext:' . preg_quote($k, '/') . ':[^】]*】/u', $v, $updated_html);
          }
        }

        // Update history (defer persisting; UI will save meta when user saves the post).
        $hist = (string) get_post_meta($post_id, 'ead_update_history', true);
        $hist_entry = "=== " . current_time('mysql') . " (update) ===
";
        $hist_entry .= "targets: " . json_encode($targets, JSON_UNESCAPED_UNICODE) . "
";
        $hist_entry .= "update_prompt:
" . (string) $update_prompt . "

";
        $hist_next = $hist ? ($hist . "
" . $hist_entry) : $hist_entry;

        return new WP_REST_Response([
          'ok' => true,
          'mode' => 'post_update',
          'updated_html' => $updated_html,
          'updates' => (array) $updates,
          'fix' => $fix_out,
          'ext' => $ext_requested,
          'history' => $hist_next,
        ], 200);
      }

      return new WP_REST_Response([
        'ok' => true,
        'updates' => $updates,
      ], 200);
},
  ]);

});
